#ifndef DATASTRUCTURES
#define DATASTRUCTURES

#include <stdlib.h>
#include <stdbool.h>

#define NONE 0
#define IS_FULL 1
#define IS_ZERO 2
#define IS_GT_ONE 3
#define IS_FALSE 4
#define IS_TRUE 5
#define NUM 8

#define CONJ 0
#define DISJ 1
#define LIT 2
#define INFEASIBLE_WITH_PATH 3
#define NOT 4

struct struct_literal {
	int queue;
	int header;
	int constraint;
};
typedef struct struct_literal literal;

struct struct_formula {
	literal lit;
	struct struct_formula * formula1;
	struct struct_formula * formula2;
	int type;
	bool freed;
	bool do_not_free;
};
typedef struct struct_formula formula;

enum xMas_TYPES {
	queue,
	merge,
	xfork,
	xswitch,
	function,
	sink,
	source,
	join,
	aswitch,
	synch,
	buffer,
};
// Structure storing a dynamically resizable array of ints
// The constant int INCREMENT is the number of elts that is added if the array is no longer able to add more elements.
struct dyn_array {
	int INCREMENT;
	int * array;
	int num_of_elts;
	int allocated;
};
struct dyn_array_array {
	struct dyn_array** array;
	int num_of_arrays;
};

struct xMas_component {
	char* id;
	int type;
	int num_of_outs;
	int* out;
	int* in_port_connected_to_out;
	int num_of_ins;
	int* in;
	int* out_port_connected_to_in;
	struct dyn_array** types;
	void** field;
	struct dyn_array** ecs;
	int num_of_ecs;
};

// A graph represented as an array of graph structs. The vertices are represented by the integers which are in the domain of this array.
// E.g.:
// struct Graph g[10];
// defines a graph with 10 vertices. Thus for each vertex there is a graph struct. Each graph struct has an arbitrary number of outgoing edges.
// The source of the edge is thus represented by its index in the array of graph structs.
// The end of the edge is represented by the integer 'neighbor'. With each edge, an arbitrary number of labels is allocated.
// The current number of labels is stored in 'num_of_labels'. The size of the space allocated the 'labels' array is represented
// in 'labels_allocated'. This allows a dynamically sized array. Similarly for the 'edges' array.
struct Edge {
	int neighbor;
	int labels_allocated;
	int num_of_labels;
	int *labels;
	int num_of_invalidated_labels;
};
struct Graph {
	struct Edge *edges;
	int num_of_edges;
	int edges_allocated;
	int * types;
	int num_of_types;
};


// Free an dyn_array
void dyn_array_init(struct dyn_array * array, int INCREMENT) {
	array->INCREMENT = INCREMENT;
	array->allocated = 0;
	array->num_of_elts = 0;
	array->array = NULL;
}
// Free an dyn_array
void dyn_array_free(struct dyn_array * array) {
	if (array->allocated >= 0) {
		free(array->array);
		array->array = NULL;
		array->num_of_elts = 0;
		array->allocated = 0;
	}
}
// Add integer `elt' as element to dyn_array.
void dyn_array_add_elt(struct dyn_array * array, int elt) {
	if (array->num_of_elts >= array->allocated) {
		array->allocated += array->INCREMENT;
		void *_tmp;
		if (array->array == NULL)
			_tmp = malloc (array->allocated * sizeof(int));
		else
			_tmp = realloc(array->array, array->allocated * sizeof(int));
		array->array = (int*)_tmp;
	}
	array->array[array->num_of_elts] = elt;
	array->num_of_elts++;
}
bool dyn_array_add_if_new(struct dyn_array* array, int elt) {
	int i;
	bool found = false;
	for (i=0;i<array->num_of_elts&&!found;i++)
		found = (array->array[i] == elt);
	if (!found)
		dyn_array_add_elt(array, elt);
	return !found;
}
bool dyn_array_member(struct dyn_array* array, int elt) {
	int i;
	bool found = false;
	for (i=0;i<array->num_of_elts&&!found;i++)
		found = (array->array[i] == elt);
	return found;
}
struct dyn_array* dyn_array_clone(struct dyn_array* array) {
	struct dyn_array * ret = (struct dyn_array*) malloc(sizeof(struct dyn_array));
	dyn_array_init(ret, array->INCREMENT);
	int i;
	for (i=0;i<array->num_of_elts;i++)
		dyn_array_add_elt(ret, array->array[i]);
	return ret;
}
void dyn_array_append_if_new(struct dyn_array* a1, struct dyn_array* a2) {
	int i;
	for (i=0;i<a2->num_of_elts;i++)
		dyn_array_add_if_new(a1, a2->array[i]);
}



// A path is a set of literals
struct struct_path {
	int num_of_elts;
	literal* elts;
};
typedef struct struct_path path;
struct set_of_paths {
	int num_of_paths;
	path** paths;
};

void path_free(path* path) {
		if (path->elts != NULL) free(path->elts);
		free(path);
}


bool* copy_array(bool* array, int size) {
		int i;
		bool *ret = (bool*) malloc (size * sizeof(bool));
		for (i=0;i<size;i++)
				ret[i] = array[i];
		return ret;
}

path* copy_path(path* p) {
	path* ret = (path*) malloc(sizeof(path));
	ret->num_of_elts = p->num_of_elts;
	ret->elts = (literal*) malloc(p->num_of_elts * sizeof(literal));
	int i;
	for (i=0;i<p->num_of_elts;i++) {
		literal lit = {p->elts[i].queue, p->elts[i].header, p->elts[i].constraint};
		ret->elts[i] = lit;
	}
	return ret;
}



typedef struct {
	literal* num;
	literal* is_full;
	literal* is_zero;
	literal* is_gt_one;
	int* vars_num;
	int* vars_is_full;
	int* vars_is_zero;
	int* vars_is_gt_one;
	int index_num;
	int index_is_full;
	int index_is_zero;
	int index_is_gt_one;
	literal* vars_to_lits;
} lits_to_vars;

typedef struct {
	int num_of_vars;
	int** clauses;
	int* clause_sizes;
	int num_of_clauses;
	int size_of_clauses;
	bool* results;
	bool results_set;
	int status;
} dimacs_instance;

#endif

