#ifndef FORMULA
#define FORMULA

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "datastructures.c"
#include "xmas_network.c"

/*	This C file contains functions for creating and freeing logical formulae.
	A formulae can be a conjunct, disjunct, negation or a literal.
	A literal can be (here NOH = NUM_OF_HEADERS):
		IS_TRUE				= true
		IS_FALSE			= false
		IS_FULL(q)			= queue q is full
		IS_ZERO(q, h)		= queue q contains no packets with header h
		IS_ZERO(q, NOH)		= queue q is empty
		IS_GT_ONE(q, h)		= queue q contains a packet with header h at the top of the queue
		IS_GT_ONE(q, NOH)	= queue q contains at least one packet

	This file contains a custom garbage collection for these formulae.
	If global variable 'garbage_collection' is true, then all created formulae are added to the garbage.
	This garbage can be emptied with a call to 'free_formulas'.
	If not, then it is up to the program to free the formulaes (note that due to mutual dependencies, this can be quite tricky).
	
	Contents:
	GARBAGE COLLECTION
	FORMULA CREATION
	PRINTING
	ACCESSING MEMOIZED FORMULA
	NORMALIZING FORMULAE TO CNF
	EXPORTING FORMULA TO SAT
 */


// The initial number of clauses allocated when creating a dimacs instance.
const int NUM_OF_CLAUSES = 5000;
// Information derived from the invariants
extern bool* type_specific_queues;
extern bool pruning_full(int curr);
//ZHCAFF extern formula* invariants;



/* GARBAGE COLLECTION */
// Each time a formula struct is allocated, the pointer is also added to the garbage. After each call of the algorithm
// the garbage is emptied.
formula** garbage = NULL;
int size_of_garbage;
int garbage_allocated = 0;
int const GARBAGE_SIZE = 100;
int const NUM_OF_POINTERS_TO_BE_FREED = 50;
bool garbage_collection = true;

void add_to_garbage(formula* f) {
	if (size_of_garbage >= garbage_allocated) {
		garbage_allocated += GARBAGE_SIZE;
		garbage = realloc(garbage, garbage_allocated * sizeof(formula*));
		if (garbage == NULL)
			printf("ERROR: increase garbage size to more than: %i\n", size_of_garbage);
	}
	garbage[size_of_garbage++] = f;
}

/* Frees all formula structures in the garbage
 */
void free_formulas(bool force) {
	formula **freed_pointers;
	int num_of_freed_pointers = 0;
	freed_pointers = malloc(NUM_OF_POINTERS_TO_BE_FREED*sizeof(formula*));
	int num_of_freed_pointers_allocated = NUM_OF_POINTERS_TO_BE_FREED;
	num_of_freed_pointers = 0;
	int i;
	for(i=0;i<size_of_garbage;i++) {
		if ((force || !garbage[i]->do_not_free) && !garbage[i]->freed) {
			garbage[i]->freed = true;
			if (num_of_freed_pointers >= num_of_freed_pointers_allocated) {
				num_of_freed_pointers_allocated += NUM_OF_POINTERS_TO_BE_FREED;
				freed_pointers = realloc(freed_pointers, num_of_freed_pointers_allocated * sizeof(formula*));
				if (freed_pointers == NULL)
					printf("ERROR: increase garbage size to more than: %i\n", size_of_garbage);
			}
			freed_pointers[num_of_freed_pointers++] = garbage[i];
		}
	}
	for (i=0;i<num_of_freed_pointers;i++)
		free(freed_pointers[i]);
	free(freed_pointers);
}


/* FORMULA CREATION */
formula* malloc_formula() {
	formula* ret = NULL;
	ret = malloc(sizeof(formula));
	ret->freed = false;
	ret->do_not_free = !garbage_collection;
	add_to_garbage(ret);
	return ret;
}

// We store the formulas true and false at one location, instead of reallocing them all the time
formula* true_formula = NULL;
formula* false_formula = NULL;

// Returns t iff the given pointer does not point to a formula
// The initial number of clauses allocated when creating a dimacs instance.
bool no_formula(formula* f1) {
	return f1 == NULL;
}

/* With each visited queue, a set of deadlock equations is memoized in formula structs.
   The following functions are used to create these formulae.
 */
/* Returns a formula containg the literal specified by the parameters.
   curr is the queue
   header is the header on which the constraint is specified.
   constraint is:
NONE:			an empty literal (should not be used as it can lead to unsoundess)
IS_FULL:		curr = curr.size;
IS_ZERO:		curr.header = 0; or curr = 0 iff header == NUM_OF_HEADERS 
IS_GT_ONE:		curr.header >= 1; or curr >= 1 iff header == NUM_OF_HEADERS
IS_FALSE:		false
IS_TRUE:		true
 */
formula* singleton_formula(int curr, int header, int constraint) {
	if (constraint == IS_FALSE) {
		if (false_formula != NULL)
			return false_formula;
		else {
			literal lit = {curr,header,constraint};
			false_formula = malloc(sizeof(formula));
			false_formula->type = LIT;
			false_formula->lit = lit;
			false_formula->freed = true;
			return false_formula;
		}
	}
	else if (constraint == IS_TRUE) {
		if (true_formula != NULL)
			return true_formula;
		else {
			literal lit = {curr,header,constraint};
			true_formula = malloc(sizeof(formula));
			true_formula->type = LIT;
			true_formula->lit = lit;
			true_formula->freed = true;
			return true_formula;
		}
	}
	formula* ret = malloc_formula();
	ret->type = LIT;
	ret->lit.queue = curr;
	ret->lit.header = header;
	ret->lit.constraint = constraint;
	return ret;
}
// Return the negation of formula f
formula* negate(formula* f) {
	formula* ret;
	if (no_formula(f))
		ret = NULL;
	else {
		ret = malloc_formula();
		ret->type = NOT;
		ret->freed = false;
		ret->formula1 = f;
	}
	return ret;
}
// Return the disjunction of formula f
formula* disjunct(formula* f1,  formula *f2) {
	formula* ret;
	if (no_formula(f1) || no_formula(f2))
		ret = NULL;
	else {
		if (f1->type == LIT && f1->lit.constraint == NONE)
			ret = f2;
		else if (f2->type == LIT && f2->lit.constraint == NONE)
			ret = f1;
		else if (f1->type == LIT && f1->lit.constraint == IS_FALSE)
			ret = f2;
		else if (f2->type == LIT && f2->lit.constraint == IS_FALSE)
			ret = f1;
		else if ((f1->type == LIT && f1->lit.constraint == IS_TRUE) || (f2->type == LIT && f2->lit.constraint == IS_TRUE))
			ret = singleton_formula(0,0, IS_TRUE);
		else {
			ret = malloc_formula();
			ret->type = DISJ;
			ret->freed = false;
			ret->formula1 = f1;
			ret->formula2 = f2;
		}
	}
	return ret;
}
// Return the conjunction of formula f
formula* conjunct(formula* f1,  formula *f2) {
	formula* ret;

	if (no_formula(f1) || no_formula(f2))
		ret = NULL;
	else {
		if (f1->type == LIT && f1->lit.constraint == NONE)
			ret = f2;
		else if (f2->type == LIT && f2->lit.constraint == NONE)
			ret = f1;
		else if (f1->type == LIT && f1->lit.constraint == IS_TRUE)
			ret = f2;
		else if (f2->type == LIT && f2->lit.constraint == IS_TRUE)
			ret = f1;
		else if ((f1->type == LIT && f1->lit.constraint == IS_FALSE) || (f2->type == LIT && f2->lit.constraint == IS_FALSE))
			ret = singleton_formula(0,0, IS_FALSE);
		else {
			ret = malloc_formula();
			ret->type = CONJ;
			ret->formula1 = f1;
			ret->formula2 = f2;
		}
	}
	return ret;
}



/* PRINTING */
void print_literal(literal lit) {
	switch (lit.constraint) {
		case NUM:
			printf("%i", lit.queue);
			break;
		case NONE:
			printf("NONE");
			break;
		case IS_FULL:
			printf("%s = %s.size", xMas_network[lit.queue].id, xMas_network[lit.queue].id);
			break;
		case IS_GT_ONE:
			if (lit.header == NUM_OF_HEADERS)
				printf("%s >= 1", xMas_network[lit.queue].id);
			else
				printf("%s.%s >= 1", xMas_network[lit.queue].id, print_header(lit.header));
			break;
		case IS_TRUE:
			printf("TRUE");
			break;
		case IS_FALSE:
			printf("FALSE");
			break;
		case IS_ZERO:
			if (lit.header == NUM_OF_HEADERS)
				printf("%s = 0", xMas_network[lit.queue].id);
			else
				printf("%s.%s = 0", xMas_network[lit.queue].id, print_header(lit.header));
			break;
		default:				
			printf("ERROR: illegal literal");
	}
}
void print_formula(formula* f1) {
	if (no_formula(f1))
		assert(false);
	switch (f1->type) {
		case CONJ:
			printf("(");
			print_formula(f1->formula1);
			printf(" AND ");
			print_formula(f1->formula2);
			printf(")");
			break;
		case DISJ:
			printf("(");
			print_formula(f1->formula1);
			printf(" OR ");
			print_formula(f1->formula2);
			printf(")");
			break;
		case NOT:
			printf("NOT(");
			print_formula(f1->formula1);
			printf(")");
			break;
		case LIT:
			print_literal(f1->lit);
			break;
		default:
			printf("ERROR: illegal formula\n");
			assert(false);
	}
}
void print_path(path* path) {
	if (path == NULL) {
		printf("NULL");
		return;
	}
	int i;
	for (i=0;i<path->num_of_elts;i++) {
		print_literal(path->elts[i]);
		printf(", ");
	}
}


/*ACCESSING MEMOIZED FORMULA*/
// Returns true iff f1 does not point to a valid formula.
// With the current struct, it is not necessary to check the formula itselve, any pointer suffices.
/* Returns t iff the formula is false
 */
bool is_formula_false(formula* f) {
	return f != NULL && f->type == LIT && f->lit.constraint == IS_FALSE;
}
bool is_formula_true(formula* f) {
	return f != NULL && f->type == LIT && f->lit.constraint == IS_TRUE;
}
// Returns t iff the two literals are equivalent
bool equal_literals(literal lit1, literal lit2) {
	if (lit1.constraint != lit2.constraint)
		return false;
	switch (lit1.constraint) {
		case NONE:
		case IS_TRUE:
		case IS_FALSE:
			return true;
		case IS_FULL:
		case NUM:
			return lit1.queue == lit2.queue;
		case IS_GT_ONE:
			return lit1.queue == lit2.queue && ((xMas_network[lit1.queue].types[0]->num_of_elts == 1 ? true : lit1.header == lit2.header) || (type_specific_queues[lit1.queue] ? lit1.header == lit2.header : true));
		case IS_ZERO:
			return lit1.queue == lit2.queue && (xMas_network[lit1.queue].types[0]->num_of_elts == 1 ? true : lit1.header == lit2.header);
		default:				
			printf("ERROR: illegal literal\n");
	}
	return false;
}
// Returns t iff the two formulae are syntactically equivalent
bool equal_formulas(formula* f1, formula* f2) {
	if (no_formula(f1) && no_formula(f2))
		return true;
	if (no_formula(f1) || no_formula(f2))
		return false;
	if (f1->type != f2->type)
		return false;
	switch (f1->type) {
		case CONJ:
		case DISJ:
			return equal_formulas(f1->formula1, f2->formula1) && equal_formulas(f1->formula2, f2->formula2);
		case NOT:
			return equal_formulas(f1->formula1, f2->formula1);
		case LIT:
			return equal_literals(f1->lit, f2->lit);
	}
	return false;
}


/*NORMALIZING FORMULAE TO CNF*/


// Flattens a formula f in cnf to a formula f_c to the following form:
// f_c	  := conjunct(literal, f_c) | f_no_c
// f_no_c := disjunct(f_no_c, f_no_c) | negate(f_no_c) | literal
formula* flatten_conjuncts(formula* f, formula* rhs) {
	if (f->type == CONJ)
		return flatten_conjuncts(f->formula1, flatten_conjuncts(f->formula2,rhs));
	else
		return conjunct(f, rhs);
}
// Returns t iff formula f is of the form:
// f_no_c := disjunct(f_no_c, f_no_c) | negate(f_no_c) | literal
bool no_conjuncts(formula* f) {
	if (no_formula(f))
		return false;
	switch (f->type) {
		case CONJ:
			return false;
		case DISJ:
			return no_conjuncts(f->formula1) && no_conjuncts(f->formula2);
		case LIT:
			return true;
		case NOT:
			return f->formula1->type == LIT;
	}
	assert(false);
	return false;
}
// Converts formula f to CNF by De Morganizing and flattening
// It uses the standard trick of using (log n) additional variables to avoid exponential explosion.
formula* to_cnf(formula* f, int* fresh) {
	if (no_formula(f))
		return NULL;
	switch (f->type) {
		case CONJ: {
			return conjunct(to_cnf(f->formula1, fresh),to_cnf(f->formula2, fresh));
		}
		case DISJ: {
			if (no_conjuncts(f->formula1)) {
				formula* rhs = flatten_conjuncts(to_cnf(f->formula2, fresh), singleton_formula(0,0,IS_TRUE));
				formula* ret = singleton_formula(0,0,IS_TRUE);
				while (rhs->type==CONJ) {
					ret = conjunct(disjunct(f->formula1, rhs->formula1), ret);
					rhs = rhs->formula2;
				}
				ret = conjunct(disjunct(f->formula1, rhs), ret);
				return ret;
			}
			else if (no_conjuncts(f->formula2)) {
				formula* rhs = flatten_conjuncts(to_cnf(f->formula1, fresh), singleton_formula(0,0,IS_TRUE));
				formula* ret = singleton_formula(0,0,IS_TRUE);
				while (rhs->type==CONJ) {
					ret = conjunct(disjunct(f->formula2, rhs->formula1), ret);
					rhs = rhs->formula2;
				}
				ret = conjunct(disjunct(f->formula2, rhs), ret);
				return ret;
			}
			else {
				formula* f_fresh = singleton_formula(*fresh, 0, NUM);
				(*fresh)++;
				formula* lhs = to_cnf(disjunct(f_fresh, f->formula1), fresh);
				formula* rhs = to_cnf(disjunct(negate(f_fresh), f->formula2), fresh);
				formula* ret = conjunct(lhs,rhs);
				return ret;
			}
			assert(false);
		}
		case NOT: {
			assert(!no_formula(f->formula1));
			formula* ret= NULL;
			switch(f->formula1->type) {
				case CONJ:
					ret = to_cnf(disjunct(negate(f->formula1->formula1),negate(f->formula1->formula2)), fresh);
					break;
				case DISJ:
					ret = to_cnf(conjunct(negate(f->formula1->formula1),negate(f->formula1->formula2)), fresh);
					break;
				case NOT:
					ret = to_cnf(f->formula1->formula1, fresh);
					break;
				case LIT:
					ret = f;
					break;
			}
			return ret;
		}
		case LIT: {
			return f;
		}
	}
	assert(false);
	return NULL;
}

// Brings formula f into normal form. For now, this is simply CNF.
formula* normalize_formula(formula* f, int* fresh) {
	return to_cnf(f, fresh);
}

// Returns t iff formula f is in CNF.
bool is_in_cnf(formula* f) {
	if (no_formula(f))
		return false;
	switch (f->type) {
		case CONJ:
			return is_in_cnf(f->formula1) && is_in_cnf(f->formula2);
		case DISJ:
			return no_conjuncts(f->formula1) && no_conjuncts(f->formula2);
		case NOT:
			return f->formula1->type == LIT;
		case LIT:
			return true;
	}
	assert(false);
	return false;
}


/*EXPORTING FORMULA TO SAT*/

// Returns a formula which returns t iff the semantics of all the literals in formula f are satisfied.
// That is:
// "q0 is full" --> "!q0 = 0"				(a full queue can not be empty)
// "q0 = 0" <--> "forall h . q0.h = 0"		(a queue is empty iff it contains no packets)
formula* semantics_of_lits(formula* f, bool* visited) {
	if (no_formula(f))
		return NULL;
	switch (f->type) {
		case CONJ:
		case DISJ: {
			formula* ret1 = semantics_of_lits(f->formula1,visited);
			formula* ret2 = semantics_of_lits(f->formula2,visited);
			if (ret1 == NULL)
				return ret2;
			if (ret2 == NULL)
				return ret1;
			return conjunct(ret1,ret2);
		}
		case NOT:
			return semantics_of_lits(f->formula1,visited);
		case LIT: {
			assert(f->lit.constraint == IS_GT_ONE ? (type_specific_queues[f->lit.queue] ? f->lit.header < NUM_OF_HEADERS : f->lit.header == NUM_OF_HEADERS) : true);
			assert(f->lit.constraint == IS_GT_ONE ? (xMas_network[f->lit.queue].types[0]->num_of_elts == 1 ? f->lit.header == NUM_OF_HEADERS : true) : true);
			assert(f->lit.constraint == IS_ZERO ? (xMas_network[f->lit.queue].types[0]->num_of_elts == 1 ? f->lit.header == NUM_OF_HEADERS : true) : true);
			assert(f->lit.constraint == IS_ZERO || f->lit.constraint == IS_FULL || f->lit.constraint == IS_GT_ONE || f->lit.constraint == NUM);

			formula* ret = NULL;
			// is_full(q) --> !is_zero(q)
			if (!visited[f->lit.queue] && f->lit.constraint != NUM && !pruning_full(f->lit.queue)) {
				ret = disjunct(negate(singleton_formula(f->lit.queue, 0, IS_FULL)), negate(singleton_formula(f->lit.queue,NUM_OF_HEADERS,IS_ZERO)));
			}
			// is_zero(q) <--> forall h . is_zero(q.h)
			if (!visited[f->lit.queue] && f->lit.constraint != NUM && xMas_network[f->lit.queue].types[0]->num_of_elts > 1) {
				formula* fc = singleton_formula(f->lit.queue, NUM_OF_HEADERS, IS_ZERO);
				formula* f1 = singleton_formula(0,0,IS_TRUE);
				int i;
				for (i=0;i<xMas_network[f->lit.queue].types[0]->num_of_elts;i++)
					f1 = conjunct(disjunct(negate(fc), singleton_formula(f->lit.queue,xMas_network[f->lit.queue].types[0]->array[i],IS_ZERO)),f1);
				assert(!is_formula_true(f1));
				formula* f2 = fc;
				for (i=0;i<xMas_network[f->lit.queue].types[0]->num_of_elts;i++)
					f2 = disjunct(negate(singleton_formula(f->lit.queue,xMas_network[f->lit.queue].types[0]->array[i],IS_ZERO)),f2);
				ret = ret == NULL ? conjunct(f1,f2) : conjunct(conjunct(f1,f2), ret);
			}

			if (f->lit.constraint != NUM)
				visited[f->lit.queue] = true;
			return ret;
		}
	}
	assert(false);
	return NULL;
}


//	Initializes a lits_to_vars struct, which is used to bookkeep which variables in the SAT instance
//	correspond to which literal.
//	Basically, it (somewhat efficiently in terms of access speed) stores a function rho of type
//	VAR_IND |--> LIT
//  where VAR_IND is the set of variable indices in the SAT instance that is constructed,
//	and LIT is the domain of literals.
lits_to_vars *init_lits_to_vars() {
	lits_to_vars* ret = (lits_to_vars*) malloc(sizeof(lits_to_vars));
	ret->num = (literal*)malloc(NUM_OF_VARS * sizeof(literal));
	ret->is_full= (literal*)malloc(NUM_OF_VARS * sizeof(literal));
	ret->is_zero = (literal*)malloc(NUM_OF_VARS * sizeof(literal));
	ret->is_gt_one = (literal*)malloc(NUM_OF_VARS * sizeof(literal));

	ret->vars_num = (int*)malloc(NUM_OF_VARS * sizeof(int));
	ret->vars_is_full= (int*)malloc(NUM_OF_VARS * sizeof(int));
	ret->vars_is_zero = (int*)malloc(NUM_OF_VARS * sizeof(int));
	ret->vars_is_gt_one = (int*)malloc(NUM_OF_VARS * sizeof(int));

	ret->vars_to_lits = (literal*)malloc(4*NUM_OF_VARS * sizeof(literal));

	ret->index_num = 0;
	ret->index_is_full= 0;
	ret->index_is_zero = 0;
	ret->index_is_gt_one = 0;
	return ret;
}
// Frees a lits_to_vars struct (see function init_lits_to_vars)
void free_lits_to_vars(lits_to_vars* ltv) {
	free(ltv->vars_to_lits);
	free(ltv->vars_is_gt_one);
	free(ltv->vars_is_zero);
	free(ltv->vars_is_full);
	free(ltv->vars_num);
	free(ltv->is_gt_one);
	free(ltv->is_zero);
	free(ltv->is_full);
	free(ltv->num);
	free(ltv);
}
// Performs the bookkeeping (see function init_lits_to_vars)
// Returns rho(lit), i.e., the index of the variable in the SAT instance corresponding to lit.
// If rho(lit) is currently undefined, then there the SAT instance has no variable representing lit.
// Function rho is updated such that rho(lit) = some fresh integer.
int lookup_in_lits_to_vars(literal lit, lits_to_vars* ltv, int* num_of_vars) {
	int i;
	int ret = -1;
	switch (lit.constraint) {
		case NUM:
			for (i=0;i<ltv->index_num&&ret==-1;i++)
				ret = equal_literals(ltv->num[i],lit) ? ltv->vars_num[i] : -1;
			if (ret == -1) {
				ret = (*num_of_vars)+1;
				if (ltv->index_num>=NUM_OF_VARS)
					printf("Set NUM_OF_VARS to a value higher than %i.\nTo do this use make with extra option `NUM_OF_VARS=x'.\n", NUM_OF_VARS);
				assert(ltv->index_num<NUM_OF_VARS);
				assert(ret < NUM_OF_VARS*4);
				(*num_of_vars)++;
				ltv->vars_num[ltv->index_num] = ret;
				ltv->num[ltv->index_num++] = lit;
				ltv->vars_to_lits[ret] = lit;
			}
			return ret;
		case IS_FULL:
			for (i=0;i<ltv->index_is_full&&ret==-1;i++)
				ret = equal_literals(ltv->is_full[i],lit) ? ltv->vars_is_full[i] : -1;
			if (ret == -1) {
				ret = (*num_of_vars)+1;
				if (ltv->index_is_full>=NUM_OF_VARS)
					printf("Set NUM_OF_VARS to a value higher than %i.\nTo do this use make with extra option `NUM_OF_VARS=x'.\n", NUM_OF_VARS);
				assert(ltv->index_is_full<NUM_OF_VARS);
				assert(ret < NUM_OF_VARS*4);
				(*num_of_vars)++;
				ltv->vars_is_full[ltv->index_is_full] = ret;
				ltv->is_full[ltv->index_is_full++] = lit;
				ltv->vars_to_lits[ret] = lit;
			}
			return ret;
		case IS_ZERO:
			//assert(type_specific_queues[lit.queue] ? true : lit.header == NUM_OF_HEADERS);
			for (i=0;i<ltv->index_is_zero&&ret==-1;i++)
				ret = equal_literals(ltv->is_zero[i],lit) ? ltv->vars_is_zero[i] : -1;
			if (ret == -1) {
				ret = (*num_of_vars)+1;
				if (ltv->index_is_zero>=NUM_OF_VARS)
					printf("Set NUM_OF_VARS to a value higher than %i.\nTo do this use make with extra option `NUM_OF_VARS=x'.\n", NUM_OF_VARS);
				assert(ltv->index_is_zero<NUM_OF_VARS);
				assert(ret < NUM_OF_VARS*4);
				(*num_of_vars)++;
				ltv->vars_is_zero[ltv->index_is_zero] = ret;
				ltv->is_zero[ltv->index_is_zero++] = lit;
				ltv->vars_to_lits[ret] = lit;
			}
			return ret;
		case IS_GT_ONE:
			assert(false);
			for (i=0;i<ltv->index_is_gt_one&&ret==-1;i++)
				ret = equal_literals(ltv->is_gt_one[i],lit) ? ltv->vars_is_gt_one[i] : -1;
			if (ret == -1) {
				ret = (*num_of_vars)+1;
				if (ltv->index_is_gt_one>=NUM_OF_VARS)
					printf("Set NUM_OF_VARS to a value higher than %i.\nTo do this use make with extra option `NUM_OF_VARS=x'.\n", NUM_OF_VARS);
				assert(ltv->index_is_gt_one<NUM_OF_VARS);
				assert(ret < NUM_OF_VARS*4);
				(*num_of_vars)++;
				ltv->vars_is_gt_one[ltv->index_is_gt_one] = ret;
				ltv->is_gt_one[ltv->index_is_gt_one++] = lit;
				ltv->vars_to_lits[ret] = lit;
			}
			return ret;
	}
	printf("Contraint = %i\n", lit.constraint);
	assert(false);
	return -1;
}

// Converts formula f into a clause. A clause is an array of variable indices.
// Assumes that f has no conjuncts.
void convert_to_clause(formula* f, int* clause, int* index_of_clause, int* num_of_vars, lits_to_vars *li) {
	if (no_formula(f))
		assert(false);
	switch (f->type) {
		case CONJ:
			assert(false);
			return;
		case DISJ:
			convert_to_clause(f->formula1, clause, index_of_clause, num_of_vars, li);
			convert_to_clause(f->formula2, clause, index_of_clause, num_of_vars, li);
			return;
		case NOT:
		case LIT: {
			formula* f1 = (f->type==NOT) ? f->formula1 : f;
			literal lit = {f1->lit.queue, f1->lit.header, f->lit.constraint};
			lit.constraint = f1->lit.constraint == IS_GT_ONE ? IS_ZERO : f1->lit.constraint;
			int var = lookup_in_lits_to_vars(lit, li, num_of_vars);
			assert(var < (*num_of_vars)+1);
			assert(equal_literals(li->vars_to_lits[var], lit));
			assert(f1->lit.constraint != NONE);
			assert(lit.constraint != NONE);
			assert(*index_of_clause < NUM_OF_VARS);
			var = f1->lit.constraint == IS_GT_ONE ? -var : var;
			var = (f->type==NOT) ? -var : var;
			clause[*index_of_clause] = var;
			(*index_of_clause)++;
			return;
		}
	}
	assert(false);
}
// Converts formula f into a set of clauses, i.e., a dimacs_instance.
// Each clause is a disjunction of variable indices.
// The lits_to_vars is used to keep track of the literals that are associated to the variable indices
// (see init_lits_to_vars).
// Assumes formula f is in cnf.
void convert_to_dimacs(formula* f, dimacs_instance* SAT, lits_to_vars* li) {
	if (no_formula(f))
		assert(false);
	switch (f->type) {
		case CONJ:
			convert_to_dimacs(f->formula1, SAT, li);
			convert_to_dimacs(f->formula2, SAT, li);
			return;
		case DISJ:
		case NOT:
		case LIT: {
			int* clause = (int*) malloc(NUM_OF_VARS * sizeof(int));
			int index_of_clause = 0;
			if (f->type == DISJ) {
				convert_to_clause(f->formula1, clause, &index_of_clause, &(SAT->num_of_vars), li);
				convert_to_clause(f->formula2, clause, &index_of_clause, &(SAT->num_of_vars), li);
			}
			else
				convert_to_clause(f, clause, &index_of_clause, &(SAT->num_of_vars), li);
			if (SAT->num_of_clauses >= SAT->size_of_clauses) {
				SAT->size_of_clauses += NUM_OF_CLAUSES;
				int** new_clauses = realloc(SAT->clauses, SAT->size_of_clauses * sizeof(int*));
				int* new_clause_sizes = realloc(SAT->clause_sizes, SAT->size_of_clauses * sizeof(int));
				assert(new_clauses != NULL);
				assert(new_clause_sizes != NULL);
				SAT->clauses = new_clauses;
				SAT->clause_sizes = new_clause_sizes;
			}
			SAT->clause_sizes[SAT->num_of_clauses] = index_of_clause;
			SAT->clauses[SAT->num_of_clauses++] = clause;
			return;
		}
	}
	assert(false);
}
// Returns the formula to check when satisfiability of formula f must be established.
// That is, it returns the conjunction of f, the invariants, and the semantics of the literals.
//ZCHAFF
/*
formula* compute_formula_for_SAT_checker(formula* f) {
	int i,fresh = 1;
	bool* visited = malloc(xMas_network_size * sizeof(bool));
	formula* f0 = conjunct(invariants, normalize_formula(f, &fresh));
	for (i=0;i<xMas_network_size;i++)
		visited[i] = false;
	formula* f1 = semantics_of_lits(f0,visited);
	if (f1 == NULL)
		f1 = f0;
	else
		f1 = conjunct(f1, f0);
	free(visited);
	return f1;
}
*/






void print_dimacs(dimacs_instance* SAT, lits_to_vars* li) {
	int i,j;
	for (i=0;i<SAT->num_of_clauses;i++) {
		for(j=0;j<SAT->clause_sizes[i];j++) {
			assert(SAT->clauses[i][j] != 0);
			literal lit = li->vars_to_lits[abs(SAT->clauses[i][j])];
			if (j>0) printf(" or ");
			if (SAT->clauses[i][j] < 0) printf("!");
			print_literal(lit);
		}
		puts("");
	}
}

dimacs_instance* init_dimacs_instance() {
	dimacs_instance* ret = (dimacs_instance*) malloc(sizeof(dimacs_instance));
	ret->clauses = (int**) malloc(NUM_OF_CLAUSES * sizeof(int*));
	ret->clause_sizes = (int*) malloc(NUM_OF_CLAUSES * sizeof(int));
	ret->num_of_clauses = 0;
	ret->size_of_clauses = NUM_OF_CLAUSES;
	ret->num_of_vars = 0;
	ret->results_set = false;
	ret->status = -1;
	ret->results = NULL;
	return ret;
}

void dimacs_free(dimacs_instance* SAT) {
	int i;
	for (i=0;i<SAT->num_of_clauses;i++)
		free(SAT->clauses[i]);
	if (SAT->results_set)
		free(SAT->results);
	free(SAT->clause_sizes);
	free(SAT->clauses);
	free(SAT);
}


#endif
