#ifndef INTERFACE_TO_Z3
#define INTERFACE_TO_Z3
#include "z3/include/z3.h"
#include "formula.c"
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>


struct Z3_invs {
	Z3_context ctx;
	Z3_ast f;
};

typedef struct Z3_invs inv_t;
inv_t* invariants;



#define LOG_Z3_CALLS

#ifdef LOG_Z3_CALLS
#define LOG_MSG(msg) Z3_append_log(msg)
#else
#define LOG_MSG(msg) ((void)0)
#endif



/**
   \brief exit gracefully in case of error.
*/
void exitf(const char* message) 
{
  fprintf(stderr,"BUG: %s.\n", message);
  exit(1);
}
/**
   \brief exit if unreachable code was reached.
*/
void unreachable() 
{
    exitf("unreachable code was reached");
}

/**
   \brief Simpler error handler.
 */
void error_handler(Z3_context c, Z3_error_code e) 
{
    printf("Error code: %d\n", e);
    exitf("incorrect use of Z3");
}
/**
   \brief Create a logical context.  

   Enable model construction. Other configuration parameters can be passed in the cfg variable.

   Also enable tracing to stderr and register custom error handler.
*/
Z3_context mk_context_custom(Z3_config cfg, Z3_error_handler err) 
{
    Z3_context ctx;
    
    Z3_set_param_value(cfg, "MODEL", "true");
    ctx = Z3_mk_context(cfg);
    Z3_set_error_handler(ctx, err);
    
    return ctx;
}
/**
   \brief Create a logical context.

   Enable model construction only.

   Also enable tracing to stderr and register standard error handler.
*/
Z3_context mk_context() 
{
    Z3_config  cfg;
    Z3_context ctx;
    cfg = Z3_mk_config();
    ctx = mk_context_custom(cfg, error_handler);
    Z3_del_config(cfg);
    return ctx;
}

/**
   \brief Prove that the constraints already asserted into the logical
   context implies the given formula.  The result of the proof is
   displayed.
   
   Z3 is a satisfiability checker. So, one can prove \c f by showing
   that <tt>(not f)</tt> is unsatisfiable.

   The context \c ctx is not modified by this function.
*/
bool is_feasible(Z3_context ctx, Z3_ast f, bool print)
{
    Z3_model m;
    /* save the current state of the context */
    Z3_push(ctx);

    Z3_assert_cnstr(ctx, f);
    
    m = 0;
	bool ret;
    
    switch (Z3_check_and_get_model(ctx, &m)) {
    case Z3_L_FALSE:
		ret = false;
        break;
    case Z3_L_UNDEF:
		ret = true;
        /* Z3 failed to prove/disprove f. */
        if (print && m != 0) {
            /* m should be viewed as a potential counterexample. */
		 Z3_string text = Z3_model_to_string(ctx, m);
		 FILE *fp = fopen(DLCONFIG_FILE, "w+");
		 if (fp != NULL) {
        	 	fputs(text, fp);
        		fclose(fp);
   	 	 }
            printf("potential counterexample:\n%s\n", text);
        }
        break;
    case Z3_L_TRUE:
        /* disproved */
		ret = true;
        if (print && m) {
            /* the model returned by Z3 is a counterexample */
		 Z3_string text = Z3_model_to_string(ctx, m);
		 FILE *fp = fopen(DLCONFIG_FILE, "w+");
		 if (fp != NULL) {
        	 	fputs(text, fp);
        		fclose(fp);
   	 	 }
            printf("Found possible config:\n%s", text);
        }
        break;
    }

    if (m) {
        Z3_del_model(ctx, m);
    }

    /* restore context */
    Z3_pop(ctx, 1);
    return ret;
}

bool equal_clauses(int* c1,int* c2, int size) {
	int i;
	bool ret = true;
	for (i=0;i<size&&ret;i++)
		ret &= c1[i] = c2[i];
	return ret;
}


Z3_ast get_Z3_rep_r(Z3_context c, Z3_ast v, char* name) {
    switch (Z3_get_ast_kind(c, v)) {
    case Z3_APP_AST: {
        unsigned i;
        Z3_app app = Z3_to_app(c, v);
        unsigned num_fields = Z3_get_app_num_args(c, app);
        Z3_func_decl d = Z3_get_app_decl(c, app);
        if (num_fields > 0) {
			Z3_ast ret = NULL;
            for (i = 0; i < num_fields&&ret==NULL; i++) {
				ret = get_Z3_rep_r(c, Z3_get_app_arg(c, app, i), name);
            }
			return ret;
        }
		else {
			int i;
			char* name2 = (char*) Z3_get_symbol_string(c, Z3_get_decl_name(c, d));
			if (!strcmp(name, name2)) {
				return v;
			}
			return NULL;
		}
        break;
    }
    default:
		break;
    }
	return NULL;
}
Z3_ast get_Z3_rep(Z3_context c, Z3_ast v, int queue, int header, bool add_if_not_existent) {
	char name[100];
	if (header >= 0 && header < NUM_OF_HEADERS)
		sprintf(name, "%s.%s", xMas_network[queue].id, print_header(header));
	else
		sprintf(name, "%s", xMas_network[queue].id);
	Z3_ast rep = get_Z3_rep_r(c, v, name);
	if (rep != NULL) return rep;
	if (!add_if_not_existent) return NULL;

	int num_args = xMas_network[queue].types[0]->num_of_elts;
	Z3_ast args[num_args];
	Z3_ast ret = NULL;
	int j;
	for (j=0;j<num_args;j++) {
		Z3_ast rep = get_Z3_rep(c, v, queue, xMas_network[queue].types[0]->array[j], false);
		if (rep == NULL) {
			char name2[100];
			sprintf(name2, "%s.%s", xMas_network[queue].id, print_header(xMas_network[queue].types[0]->array[j]));
			rep = Z3_mk_const(c, Z3_mk_string_symbol(c, name2), Z3_mk_int_sort(c));
		}
		args[j] = rep;
		if (xMas_network[queue].types[0]->array[j]==header) ret = rep;
	}
	Z3_ast q = get_Z3_rep(c, v, queue, NUM_OF_HEADERS, false);
	if (q == NULL) {
		char nameq[100];
		sprintf(nameq, "%s", xMas_network[queue].id);
		q = Z3_mk_const(c, Z3_mk_string_symbol(c, name), Z3_mk_int_sort(c));
		int qsize = ((int*) (xMas_network[queue].field[0]))[0];	
		Z3_assert_cnstr(c, Z3_mk_le(c, q, Z3_mk_int(c, qsize, Z3_mk_int_sort(c))));
	}
	if (header == NUM_OF_HEADERS || num_args == 0)
		ret = q;
	if (num_args != 0)
		Z3_assert_cnstr(c, Z3_mk_eq(c, q, Z3_mk_add(c, num_args, args)));
	else
		Z3_assert_cnstr(c, Z3_mk_eq(c, q, Z3_mk_int(c, 0, Z3_mk_int_sort(c))));
		
	return ret;
}

Z3_ast convert_literal_to_Z3(literal lit) {
	Z3_context c = invariants->ctx;
	switch (lit.constraint) {
		case NUM: {
			Z3_sort int_ty = Z3_mk_int_sort(c);
			return Z3_mk_int(c, 0, int_ty);
			printf("%i", lit.queue);
			break;
		}
		case NONE:
			break;
		case IS_FULL: {
			Z3_sort int_sort = Z3_mk_int_sort(c);
			int qsize = ((int*) (xMas_network[lit.queue].field[0]))[0];	
			return Z3_mk_eq(c, get_Z3_rep(c, invariants->f, lit.queue, NUM_OF_HEADERS, true), Z3_mk_int(c, qsize, int_sort)); 
			break;
		}
		case IS_GT_ONE: {
			Z3_sort int_sort = Z3_mk_int_sort(c);
			Z3_ast z3_rep = get_Z3_rep(c, invariants->f, lit.queue, lit.header, true);
			return Z3_mk_ge(c, z3_rep, Z3_mk_int(c, 1, int_sort));
		}
		case IS_TRUE:
			printf("TRUE");
			break;
		case IS_FALSE:
			printf("FALSE");
			break;
		case IS_ZERO: {
			Z3_sort int_sort = Z3_mk_int_sort(c);
			Z3_ast z3_rep = get_Z3_rep(c, invariants->f, lit.queue, lit.header, true);
			return Z3_mk_eq(c, z3_rep, Z3_mk_int(c, 0, int_sort));
		}
		default:				
			printf("ERROR: illegal literal");
	}
}
Z3_ast convert_formula_to_Z3(formula* f) {
	Z3_context c = invariants->ctx;
	if (no_formula(f))
		assert(false);
	switch (f->type) {
		case CONJ: {
			Z3_ast args[2];
			args[0] = convert_formula_to_Z3(f->formula1);
			args[1] = convert_formula_to_Z3(f->formula2);
			return Z3_mk_and(c, 2, args);
			break;
		}
		case DISJ: {
			Z3_ast args[2];
			args[0] = convert_formula_to_Z3(f->formula1);
			args[1] = convert_formula_to_Z3(f->formula2);
			return Z3_mk_or(c, 2, args);
			break;
		}
		case NOT:
			return Z3_mk_not(c, convert_formula_to_Z3(f->formula1));
			break;
		case LIT:
			return convert_literal_to_Z3(f->lit);
			break;
		default:
			printf("ERROR: illegal formula\n");
			assert(false);
	}
}

void assert_queue_size(inv_t* inv, int queue) {
	Z3_sort int_sort = Z3_mk_int_sort(inv->ctx);
	int qsize = ((int*) (xMas_network[queue].field[0]))[0];	
	Z3_ast z3_f = Z3_mk_le(inv->ctx, get_Z3_rep(inv->ctx, inv->f, queue, NUM_OF_HEADERS, true), Z3_mk_int(inv->ctx, qsize, int_sort)); 
//printf ("Asserting: ");printf("%s\n", Z3_ast_to_string(inv->ctx, z3_f));
	Z3_assert_cnstr(inv->ctx, z3_f);
}

// Returns t iff formula f is satisfiable.
bool check_feasibility_formula(formula* f, bool print) {
	// See if the formual is trivial
	if (is_formula_true(f))
		return true;
	if (is_formula_false(f))
		return false;

	Z3_ast z3_f = convert_formula_to_Z3(f);
	return is_feasible(invariants->ctx, z3_f, print);
	
	assert(false);
	return false;
}
#endif
