#include <stdio.h>
#include <stdbool.h>
#include <time.h>
#include "paths.c"
#include "simulation.c"
#include "liveness_propagation.c"

//#include "export_to_svg.c"

/*
   Deadlock freedom decision procedure for xMas networks

   Functions perm_dead en perm_idle contain the algorithm and further explanation.
   Function main maps everything together.
 */




//-----DATA STRUCTURES-----//
// We can mark each queue as visited or not by perm_block.
bool* block_visited;
// We can mark each queue/header as visited or not by perm_idle.
bool** idle_visited;

// As soon as a deadlock has been detected, the following variable is set to true.
bool found_deadlock = false;

// We keep track of the set of equations corresponding to the path leading from the initial queue to the current component.
// These equations consists of:
// 1. queues that are full, i.e., q0 = q0.size. This is stored as: eqns_full[q0] = true;
bool* eqns_full;
// 2. queues in which there is a packet, i.e., q0.h >= 1. This is stored as eqns_gt1[q0] = h.
// The equation q0 >= 1 is stored as eqns_gt1[q0] = NUM_OF_HEADERS 
int* eqns_gt1;
// 3. queues are empty, i.e., q0 = 0. This is stored as: eqns_zero[q0] = true
bool** eqns_zero;
// The exact same information is also stored as a path, i.e, a sequence of equations:
path* curr_path;
// We apply an optimization trick: for networks without joins, function perm_idle is never called.
// The following variable is set to true as soon as perm_idle is called. Only if this has happened, the
// array `idle_visited' is cleaned after each run of the algorithm.
bool idle_equation_set = false;

// For each queue, we keep track of the paths leading to the queue for which the SAT solver has already
// been called. If the algorithm arrives at a visited queue q, and the current path is subsumed by one of
// the paths in paths[q], the algorithm does not call the SAT solver again.
struct set_of_paths** bl_paths;
struct set_of_paths*** id_paths;
formula*** bl_formula;
formula**** id_formula;
// For each queue q of which deadlock freedom has been established definitely, sure[q] is set to true.
bool *sure;

// At least the maximum size of the paths:
const int PATH_SIZE = 2000;
// A value representing at least the number of paths that are memoized per queue
int const NUM_OF_PATHS_PER_QUEUE = 20000;

// Iff the invariants state !is_full(q) then pruning_full(q) returns true.
extern bool* pruning_is_full;




formula* perm_idle(int curr, int header, int out_port, int open_edges, bool do_add_eqns_empty);
formula* perm_blocking(int curr, int header, int in_port, int open_edges, int depth, bool do_add_eqns_full);

formula* compute_dl_eq(int curr, int h, int out_port, int open_edges, bool do_add_eqns_full, int depth) {
	formula* ret = perm_blocking(xMas_network[curr].out[out_port], h, xMas_network[curr].in_port_connected_to_out[out_port],open_edges,depth,do_add_eqns_full);
	return ret;
}
formula* compute_id_eq(int curr, int h, int in_port, int open_edges) {
	int h_new = h; //xMas_network[xMas_network[curr].in[in_port]].type == queue || xMas_network[xMas_network[curr].in[in_port]].type == buffer ? (xMas_network[xMas_network[curr].in[in_port]].types[0]->num_of_elts == 1 ? NUM_OF_HEADERS : h) : h;
	formula* ret = perm_idle(xMas_network[curr].in[in_port], h_new, xMas_network[curr].out_port_connected_to_in[in_port],open_edges, true);
	return ret;
}


formula*  propagate_packet_at_channel (int curr, int header, int out_port) {
//printf("PPAC: %s.%s \n", xMas_network[curr].id, print_header(header));
	///if (sure[curr]) return singleton_formula(0,0,IS_FALSE);
	if (curr == -1 || (xMas_network[curr].type != source && xMas_network[curr].in[0] == -1)) return singleton_formula(0,0,IS_FALSE);
	if (out_port == -1) return singleton_formula(0,0,IS_FALSE);
	switch (xMas_network[curr].type) {
		case buffer:
		case queue:
			return singleton_formula(curr,header,IS_GT_ONE);
		case function: {
			formula* ret = singleton_formula(0,0,IS_FALSE);
			if (header != NUM_OF_HEADERS) {
				int i;
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts;i++) {
					int old_header = xMas_network[curr].types[0]->array[i];
					if ( (((int (*)(int)) (xMas_network[curr].field[0]))(old_header)) == header)
						ret = disjunct(ret, propagate_packet_at_channel (xMas_network[curr].in[0], old_header, xMas_network[curr].out_port_connected_to_in[0]));
				}
			}
			else
				ret = propagate_packet_at_channel (xMas_network[curr].in[0], NUM_OF_HEADERS, xMas_network[curr].out_port_connected_to_in[0]);
			return ret;
		}
		case source: {
			int i;
			int num_of_packets = ((int*)(xMas_network[curr].field[0]))[0];
			bool found = (header == NUM_OF_HEADERS && num_of_packets != 0);
			for (i=1;i<num_of_packets+1&&!found;i++)
				found = ((((int*) (xMas_network[curr].field[0]))[i]) == header);
			if (found)
				return singleton_formula(0, 0, IS_TRUE);
			else
				return singleton_formula(0, 0, IS_FALSE);
		}
		case xswitch:
		case aswitch:
			if (header != NUM_OF_HEADERS) {
				bool b = ((int (*)(int)) (xMas_network[curr].field[0]))(header);
				if (b == (out_port == 0))
					return propagate_packet_at_channel (xMas_network[curr].in[0], header, xMas_network[curr].out_port_connected_to_in[0]);
			}
			else {
				int i, new_header;
				bool b;
				formula* ret = singleton_formula(0,0,IS_FALSE);
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts;i++) {
					new_header = xMas_network[curr].types[0]->array[i];
					b = ((int (*)(int)) (xMas_network[curr].field[0]))(new_header);
					if (b == (out_port == 0))
						ret = disjunct(ret, propagate_packet_at_channel (xMas_network[curr].in[0], header, xMas_network[curr].out_port_connected_to_in[0]));
				}
				return ret;
			}
			return;
		case xfork:
		case synch:
		case merge: {
			int in;
			formula* ret = singleton_formula(0,0,IS_FALSE);
			for (in=0;in<xMas_network[curr].num_of_ins;in++)
				ret = disjunct(ret, propagate_packet_at_channel (xMas_network[curr].in[in], header, xMas_network[curr].out_port_connected_to_in[in]));
			return;
		}
		case join: {
			int token_input = ((int*) (xMas_network[curr].field[0]))[0];
			int data_input = token_input == 0 ? 1 : 0;
			return conjunct(propagate_packet_at_channel (xMas_network[curr].in[data_input], header, xMas_network[curr].out_port_connected_to_in[data_input]),
						propagate_packet_at_channel (xMas_network[curr].in[token_input], NUM_OF_HEADERS, xMas_network[curr].out_port_connected_to_in[token_input]));
		}
	}
	printf("ERROR: unknown component\n");
	assert(false);
}





//----THE ALGORITHM-----//
formula* perm_blocking(int curr, int header, int in_port, int open_edges, int depth, bool do_add_eqns_full) {
	if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
	if (sure[curr])	return singleton_formula(0,0,IS_FALSE);
//if (xMas_network[curr].type == queue || xMas_network[curr].type == buffer)  printf("DL: %s.%s (open-edges == %i, %i)\n", xMas_network[curr].id, print_header(header), open_edges, curr_path->num_of_elts);
	switch (xMas_network[curr].type) {
		case buffer:
		case queue: {
			/* If the current queue can never be full, we can simply return false */
			if ((do_add_eqns_full && pruning_full(curr)))
				return singleton_formula(0,0,IS_FALSE);

			formula* ret = NULL;
			int i,new_header;
			int subsumed = subsumed_by_earlier_bl_visit(curr_path, curr);
			if (block_visited[curr] == false && subsumed==-1) {
				/* If the current queue has not been visited and the current path is not subsumed by earlier visits: */
				/* Add the constraint that the current queue is full, as it must be blocking */
				bool backup = eqns_full[curr];
				int id = -1;
				if (!do_add_eqns_full || ((id = add_eqns_full_queue(curr)) != -1)) {
					/* Mark the current queue as visited */
					block_visited[curr] = true;
					/* Compute deadlock equation of current component */
					formula* ret2 = singleton_formula(curr, header, IS_FALSE);
					for (i=0;i<xMas_network[curr].ecs[0]->num_of_elts;i++) {
						new_header = xMas_network[curr].ecs[0]->array[i];
						int id2 = -1;
						if ((id2 = add_eqns_packet_in_queue(curr, new_header)) != -1) {
							//if (check_consistency_path2()) {//Irrelevant for correctness, but can have significant impact on running time
							formula* ret3 = perm_blocking(xMas_network[curr].out[0], new_header, xMas_network[curr].in_port_connected_to_out[0],open_edges, depth+1, true);
							if (type_specific_queues[curr])
								ret3 = conjunct(ret3, singleton_formula(curr, new_header, IS_GT_ONE));
							ret2 = disjunct(ret2, ret3);
							if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
							//}
							retract_curr_path(id2);
							eqns_gt1[curr] = -1;
						}
					}
					if (do_add_eqns_full) 
						ret2 = conjunct(singleton_formula(curr, 0, IS_FULL), ret2);
					else if (!type_specific_queues[curr])
						ret2 = conjunct(singleton_formula(curr, NUM_OF_HEADERS, IS_GT_ONE), ret2);


					//printf("DL_EQ of %s established (open_edges == %i, curr == %i)\n", xMas_network[curr].id, open_edges, curr);
					//print_formula(ret2);puts("");

					/* Mark the queue as unvisited */
					block_visited[curr] = false;
					if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
					/* Remove the constraint that the current queue is full */
					if (do_add_eqns_full) retract_curr_path(id);
					eqns_full[curr] = backup;
					/* If the blocking formula is implied by the current path it is trivially true */
					if (formula_subsumed_by_path(ret2, curr_path)) {
						ret = singleton_formula(0,0,IS_TRUE);
						store_bl_formula_for_path(curr, curr_path, ret);
						return ret;
					}
					else {
						store_bl_formula_for_path(curr, curr_path, ret2);
						return ret2;
					}
				}
				else {
					/* If the current queue must be full, but cannot be full return false */
					ret = singleton_formula(0,0,IS_FALSE);
					store_bl_formula_for_path(curr, curr_path, ret);
					return ret;
				}
			}
			else if (pruning_full(curr)) {
				/* If the current queue must be full, but cannot be full return false */
				return singleton_formula(0,0,IS_FALSE);
			}
			else if (block_visited[curr] == true && subsumed==-1) {
				/* If the current queue has already been visited and the current path is not subsumed by an earlier visit: */
				bool backup = eqns_full[curr];
				/* If the current queue can be full */
				int id = -1;
				if (!do_add_eqns_full || ((id = add_eqns_full_queue(curr)) != -1)) { 
					bool feasible = pruning_full(curr) ? false : check_consistency_path2();
					if (open_edges == 0 && feasible) {
						found_deadlock = true;
						//print_eqns();
						bool b = check_feasibility_formula(curr_path_to_formula(), true);
						ret = singleton_formula(0,0,IS_TRUE);
						return ret;
					}
					if (do_add_eqns_full) retract_curr_path(id);
					eqns_full[curr] = backup;
					if (feasible) {
						ret = eqns_full[curr] ? singleton_formula(curr,0,IS_TRUE) : singleton_formula(curr,0,IS_FULL);
						store_bl_formula_for_path(curr, curr_path, ret);
						return ret;
					}
				}
				eqns_full[curr] = backup;
				ret = singleton_formula(0,0,IS_FALSE);
				store_bl_formula_for_path(curr, curr_path, ret);
				return ret;
			}
			else if (subsumed != -1) {
				//printf("Subsumed bl, returning\n");
				//print_formula(bl_formula[curr][subsumed]);
				return bl_formula[curr][subsumed];
			}
			assert(false);
			return NULL;
		}
		case function: {
			int next_queue; 
			if (header != NUM_OF_HEADERS) {
				int new_header = ((int (*)(int)) (xMas_network[curr].field[0]))(header);
				return perm_blocking(xMas_network[curr].out[0], new_header, xMas_network[curr].in_port_connected_to_out[0],open_edges,depth+1, true);
			}
			else {
				return perm_blocking(xMas_network[curr].out[0], header, xMas_network[curr].in_port_connected_to_out[0],open_edges,depth+1, true);
			}
		}
		case source:
			return perm_blocking(xMas_network[curr].out[0], header, xMas_network[curr].in_port_connected_to_out[0],open_edges,depth, true);
		case sink: {
			return singleton_formula(0,0,IS_FALSE);
			break;
		}
		case xfork: {
			formula* ret1 = compute_dl_eq(curr, header, 0, open_edges, true, depth+1);
			if (is_formula_true(ret1)) return ret1;
			formula* ret2 = compute_dl_eq(curr, header, 1, open_edges, true, depth+1);
			return disjunct(ret1, ret2);
		}
		case join: {
			int token_input = ((int*) (xMas_network[curr].field[0]))[0];
			formula* ret1;
			if (in_port == token_input)
				ret1 = compute_dl_eq(curr, NUM_OF_HEADERS, 0, open_edges, true, depth+1);
			else 
				ret1 = compute_dl_eq(curr, header, 0, open_edges, true, depth+1);
			if (is_formula_true(ret1)) return ret1;
			formula* ret2 = compute_id_eq(curr, NUM_OF_HEADERS, (in_port == 0) ? 1 : 0, open_edges);
			return disjunct(ret1,ret2);
		}
		case xswitch: {
			if (header != NUM_OF_HEADERS) {
				bool b = ((int (*)(int)) (xMas_network[curr].field[0]))(header);
				if (b)
					return compute_dl_eq(curr, header, 0, open_edges, true, depth+1);
				else 
					return compute_dl_eq(curr, header, 1, open_edges, true, depth+1);
			}
			else {
				int h,i;
				bool b;
				formula* ret = singleton_formula(0,0,IS_FALSE);
				for (i=0;i<xMas_network[curr].ecs[0]->num_of_elts;i++) {
					h = xMas_network[curr].ecs[0]->array[i];
					b = ((int (*)(int)) (xMas_network[curr].field[0]))(h);
					ret = disjunct(ret, b ? compute_dl_eq(curr, h, 0, open_edges, true, depth+1) : compute_dl_eq(curr, h, 1, open_edges, true, depth+1));
				}
				return ret;
			}
			break;
		}
		case merge: {
			formula* ret1 = compute_dl_eq(curr, header, 0, open_edges, true, depth+1);
			int other_in = in_port == 0 ? 1 : 0, i;
			if (!is_formula_true(ret1)) {
				formula* ret2 = singleton_formula(0,0,IS_FALSE);
				for (i=0;i<xMas_network[curr].types[other_in]->num_of_elts;i++) {//USE ECs instead of types
					formula* ret3 = compute_dl_eq(curr, xMas_network[curr].types[other_in]->array[i], 0, open_edges+1, true, depth+1);
					if (!is_formula_false(ret3)) {
						ret2 = disjunct(ret2, conjunct(ret3, propagate_packet_at_channel (xMas_network[curr].in[other_in], xMas_network[curr].types[other_in]->array[i], xMas_network[curr].out_port_connected_to_in[other_in])));
					}
				}
				ret1 = disjunct(ret1, ret2);
			}
			return ret1;
			break;
		}
		case aswitch: {
			int i,iter = 0;
			int h;
			formula* ret = singleton_formula(0,0,IS_FALSE);
			int var1;
			formula* ret7;
			int ret11;
			int ret12;
			formula* ret8;
			formula* ret5;
			int ret17;
			formula* ret14;
			formula* ret2;
			formula* ret0;
			for (i=0;i<xMas_network[curr].ecs[0]->num_of_elts && (header == NUM_OF_HEADERS || iter==0);i++) {
				iter++;
				if (header != NUM_OF_HEADERS)
					h = header;
				else
					h = xMas_network[curr].ecs[0]->array[i];
				ret0 = singleton_formula(0, 0, IS_TRUE);
				for (var1=0;var1<xMas_network[curr].num_of_outs&&!is_formula_false(ret0);var1++) {
					ret7 = singleton_formula(0, 0, IS_FALSE);
					if (!is_formula_true(ret7)) {
						ret11 = h;
						ret12 = ((int (*)(int)) (xMas_network[curr].field[var1]))(ret11);
						ret8 = ret12 ? singleton_formula(0,0,IS_FALSE) : singleton_formula(0,0,IS_TRUE);
						ret5 = disjunct(ret7,ret8);
					}
					else
						ret5 = ret7;
					if (!is_formula_true(ret5)) {
						ret17 = h;
						ret14 = compute_dl_eq(curr, ret17, var1, open_edges+1,true, depth+1);
						ret2 = disjunct(ret5,ret14);
					}
					else
						ret2 = ret5;
					ret0 = conjunct(ret0, ret2);
				}
				ret = disjunct(ret, ret0);
			}
			return ret;
		}
		case synch: {
			formula* ret = singleton_formula(0,0,IS_FALSE);
			int in,out;
			for (out=0;out<xMas_network[curr].num_of_outs&&!is_formula_true(ret);out++) {
				formula* ret1;
				if (out == in_port)
					ret1 = compute_dl_eq(curr,header,out,open_edges,true, 0);
				else
					ret1 = compute_dl_eq(curr,NUM_OF_HEADERS,out,open_edges,true,0);
				ret = disjunct(ret,ret1);
			}
			if (is_formula_true(ret)) return ret;
			for (in=0;in<xMas_network[curr].num_of_ins&&!is_formula_true(ret);in++) {
				formula* ret1;
				if (in != in_port) {
					ret1 = compute_id_eq(curr,NUM_OF_HEADERS,in,open_edges);
					ret = disjunct(ret,ret1);
				}
			}
			return ret;
		}
	}
	printf("ERROR: unknown component\n");
	assert(false);
	return NULL;
}




formula* perm_idle(int curr, int header, int out_port, int open_edges, bool do_add_eqns_empty) {
	if (sure[curr]) return singleton_formula(0,0,IS_FALSE);
	if (curr == -1 || (xMas_network[curr].type != source && xMas_network[curr].in[0] == -1)) {
		assert(false);
		printf("Not doing backwards checking, as this not necessary.\n");
		return singleton_formula(0,0,IS_FALSE);
	}
	idle_equation_set = true;
	if (idle_visited[curr] == NULL) {
		if (id_paths[curr] == NULL) {
			id_paths[curr] = malloc((NUM_OF_HEADERS+1) * sizeof(struct set_of_paths*));
			id_formula[curr] = malloc((NUM_OF_HEADERS+1) * sizeof(formula**));
			int j;
			for (j=0;j<=NUM_OF_HEADERS;j++) {
				id_paths[curr][j] = malloc(sizeof(struct set_of_paths));
				id_paths[curr][j]->num_of_paths = 0;
				id_paths[curr][j]->paths = malloc(NUM_OF_PATHS_PER_QUEUE * sizeof(path*));
				id_formula[curr][j] = malloc(NUM_OF_PATHS_PER_QUEUE * sizeof(formula*));
			}
		}
		idle_visited[curr] = calloc(NUM_OF_HEADERS+1, sizeof(bool));
	}
	//if (xMas_network[curr].type == queue || xMas_network[curr].type == buffer) printf("ID: %s.%s (out = %i), fd = %i\n", xMas_network[curr].id, print_header(header), out_port, found_deadlock);
	if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
	switch (xMas_network[curr].type) {
		case buffer:
		case queue: {
			path* path1 = curr_path;
			int subsumed = subsumed_by_earlier_id_visit(path1, curr, header);
			formula* ret = NULL;
			if (idle_visited[curr][header] == false && subsumed==-1) {
				bool found = (header == NUM_OF_HEADERS);
				int i;
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts&&!found;i++)
					found = (xMas_network[curr].types[0]->array[i] == header);
				if (found) {
					bool backup = eqns_zero[curr][header];
					int id = -1;
					if (!do_add_eqns_empty || ((id = add_eqns_empty_queue(curr,header)) != -1)) {
						idle_visited[curr][header] = true;
						if (check_consistency_path2()) {
							formula* ret1 = compute_id_eq(curr,header,0,open_edges);
							ret1 = conjunct(singleton_formula(curr, header, IS_ZERO), ret1);
							formula* ret2 = singleton_formula(0,0,IS_FALSE);
							if (xMas_network[curr].type != buffer && header != NUM_OF_HEADERS) {
								//if (do_add_eqns_empty) {
								//	retract_curr_path(id);
								//	eqns_zero[curr][header] = backup;
								//}
								ret2 = perm_blocking(curr, NUM_OF_HEADERS, 0, open_edges, 0, false);
								//if (do_add_eqns_empty) id = add_eqns_empty_queue(curr,header);
								if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
							}
							ret = disjunct(ret1, ret2);
						}
						else {
							if (do_add_eqns_empty) retract_curr_path(id);
							eqns_zero[curr][header] = backup;
							idle_visited[curr][header] = false;
							ret = singleton_formula(0,0,IS_FALSE);
							store_id_formula_for_path(curr, header, path1, ret);
							return ret;
						}
						if (do_add_eqns_empty) retract_curr_path(id);
						eqns_zero[curr][header] = backup;
						idle_visited[curr][header] = false;

						//printf("ID_EQ of %s established (open_edges == %i, curr == %i, header = %s)\n", xMas_network[curr].id, open_edges, curr, print_header(header));
						//print_formula(ret);puts("");
						//printf("For path: ");print_path(curr_path);puts("");
						if (formula_subsumed_by_path(ret, path1)) {
							ret = singleton_formula(0,0,IS_TRUE);
							store_id_formula_for_path(curr, header, path1, ret);
						}
						else {
							store_id_formula_for_path(curr, header, path1, ret);
						}
						return ret;
					}
					else {
						ret = singleton_formula(0,0,IS_FALSE);
						store_id_formula_for_path(curr, header, path1, ret);
						return ret;
					}
				}
				else {
					ret = singleton_formula(0,0,IS_TRUE);
					store_id_formula_for_path(curr, header, path1, ret);
					return ret;
				}
			}
			else if (idle_visited[curr][header] == true && subsumed==-1) {
				bool backup = eqns_zero[curr][header];
				/* If the current queue can be empty */
				int id = -1;
				if (!do_add_eqns_empty || ((id = add_eqns_empty_queue(curr,header)) != -1)) { 
					bool feasible = check_consistency_path2();
					if (open_edges == 0 && feasible) {
						found_deadlock = true;
						//print_eqns();
						check_feasibility_formula(curr_path_to_formula(), true);
						ret = singleton_formula(0,0,IS_TRUE);
						return ret;
					}
					if (do_add_eqns_empty) retract_curr_path(id);
					eqns_zero[curr][header] = backup;
					if (feasible) {
						ret = eqns_zero[curr][header] ? singleton_formula(0,0,IS_TRUE) : singleton_formula(curr,header,IS_ZERO);
						store_id_formula_for_path(curr, header, path1, ret);
						return ret;
					}
				}
				eqns_zero[curr][header] = backup;
				ret = singleton_formula(0,0,IS_FALSE);
				store_id_formula_for_path(curr, header, path1, ret);
				return ret;
			}
			else if (subsumed != -1) {
				//printf("Subsumed id, returning\n");
				//print_formula(id_formula[curr][header][subsumed]);
				return id_formula[curr][header][subsumed];
			}
			assert(false);
		}
		case function:
			if (header != NUM_OF_HEADERS) {
				formula* ret = singleton_formula(0,0,IS_TRUE);
				int i;
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts&&!is_formula_false(ret);i++) {
					int old_header = xMas_network[curr].types[0]->array[i];
					if ( (((int (*)(int)) (xMas_network[curr].field[0]))(old_header)) == header) {
						ret = conjunct(ret, compute_id_eq(curr,old_header,0,open_edges+1));
					}
				}
				return ret;
			}
			else {
				formula* ret = compute_id_eq(curr,header,0,open_edges);
				return ret;
			}
		case source: {
			int i;
			int num_of_packets = ((int*)(xMas_network[curr].field[0]))[0];
			bool found = (header == NUM_OF_HEADERS && num_of_packets != 0);
			for (i=1;i<num_of_packets+1&&!found;i++)
				found = ((((int*) (xMas_network[curr].field[0]))[i]) == header);
			if (found)
				return singleton_formula(0, 0, IS_FALSE);
			else {
				if (open_edges == 0) {
					//TODO
				}
				return singleton_formula(0, 0, IS_TRUE);
			}
			assert(false);
		}
		case sink: {
			printf("ERROR\n");
			break;
		}
		case xfork: {
			formula* ret1 = compute_dl_eq(curr,header,(out_port == 0) ? 1 : 0,open_edges,true, 0);
			if (is_formula_true(ret1)) return ret1;
			if (found_deadlock) return singleton_formula(0,0,IS_TRUE);
			formula* ret2 = compute_id_eq(curr,header,0,open_edges);
			return disjunct(ret1, ret2);
		}
		case join: {
			int token_input = ((int*) (xMas_network[curr].field[0]))[0];
			int data_input = token_input == 0 ? 1 : 0;
			formula* ret1 = compute_id_eq(curr, header, data_input, open_edges);
			if (is_formula_true(ret1)) return ret1;
			formula* ret2 = compute_id_eq(curr, NUM_OF_HEADERS, token_input, open_edges);
			return disjunct(ret1, ret2);
		}
		case xswitch:
			if (header != NUM_OF_HEADERS) {
				bool b = ((int (*)(int)) (xMas_network[curr].field[0]))(header);
				if (b == (out_port == 0)) {
					formula* ret2 = compute_id_eq(curr,header,0,open_edges);
					return ret2;
				}
				else {
					if (open_edges == 0) {
						printf("HALLO3\n");assert(false);
					}
					return singleton_formula(curr, header, IS_TRUE);
				}
			}
			else {
				int i, new_header;
				bool b;
				formula* ret1 = singleton_formula(curr, header, IS_TRUE);
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts&&!is_formula_false(ret1);i++) {
					new_header = xMas_network[curr].types[0]->array[i];
					b = ((int (*)(int)) (xMas_network[curr].field[0]))(new_header);
					if (b == (out_port == 0)) {
						formula* ret2 = compute_id_eq(curr,new_header,0,open_edges+1);
						ret1 = conjunct(ret1, ret2);
					}
				}
				return ret1;
			}
			break;
		case aswitch: {
			if (header != NUM_OF_HEADERS) {
				bool b = ((int (*)(int)) (xMas_network[curr].field[out_port]))(header);
				if (b) {
					formula* ret2 = compute_id_eq(curr,header,0,open_edges);
					return ret2;
				}
				else {
					if (open_edges == 0) {
						printf("HALLO3\n");assert(false);
					}
					return singleton_formula(curr, header, IS_TRUE);
				}
			}
			else {
				int i, new_header;
				bool b;
				formula* ret1 = singleton_formula(curr, header, IS_TRUE);
				for (i=0;i<xMas_network[curr].types[0]->num_of_elts&&!is_formula_false(ret1);i++) {
					new_header = xMas_network[curr].types[0]->array[i];
					b = ((int (*)(int)) (xMas_network[curr].field[out_port]))(new_header);
					if (b) {
						formula* ret2 = compute_id_eq(curr,new_header,0,open_edges+1);
						ret1 = conjunct(ret1, ret2);
					}
				}
				return ret1;
			}
		}
		case merge: {
			int in;
			formula* ret = singleton_formula(0,0,IS_TRUE);
			for (in=0;in<xMas_network[curr].num_of_ins&&!is_formula_false(ret);in++) {
				formula* ret1 = compute_id_eq(curr, header, in, open_edges+1);
				ret = conjunct(ret, ret1);
			}
			return ret;
		}
		case synch: {
			formula* ret = singleton_formula(0,0,IS_FALSE);
			int in,out;
			for (out=0;out<xMas_network[curr].num_of_outs&&!is_formula_true(ret);out++) {
				if (out != out_port) {
					formula* ret1 = compute_dl_eq(curr,NUM_OF_HEADERS,out,open_edges,true,0);
					ret = disjunct(ret,ret1);
				}
			}
			if (is_formula_true(ret)) return ret;
			for (in=0;in<xMas_network[curr].num_of_ins&&!is_formula_true(ret);in++) {
				formula* ret1;
				if (in == out_port)
					ret1 = compute_id_eq(curr,header,in,open_edges);
				else
					ret1 = compute_id_eq(curr,NUM_OF_HEADERS,in,open_edges);
				ret = disjunct(ret,ret1);
			}
			return ret;
		}
	}
	assert(false);
	return NULL;
}




//-----MAIN FUNCTION----//
int main (void) {
	// Load the network into memory
	printf("Loading network...\n");
	init_xMas_network();


	// Initialize all data structures
	int i,j,k,l,flg;
	eqns_full = malloc(xMas_network_size * sizeof(bool));
	eqns_gt1  = malloc(xMas_network_size * sizeof(int));
	eqns_zero = malloc(xMas_network_size * sizeof(bool*));
	block_visited = malloc(xMas_network_size * sizeof(bool));
	idle_visited = malloc(xMas_network_size * sizeof(bool*));
	bl_paths = malloc(xMas_network_size * sizeof(struct set_of_paths*));
	id_paths = malloc(xMas_network_size * sizeof(struct set_of_paths**));
	bl_formula = malloc(xMas_network_size * sizeof(formula**));
	id_formula = malloc(xMas_network_size * sizeof(formula***));
	size_of_garbage = 0;
	curr_path = malloc(sizeof(path));
	curr_path->num_of_elts = 0;
	curr_path->elts = malloc(PATH_SIZE * sizeof(literal));

	for (i=0;i<xMas_network_size;i++) {
		bl_paths[i] = NULL;
		id_paths[i] = NULL;
		if (xMas_network[i].type == queue || xMas_network[i].type == buffer) {
			bl_paths[i] = malloc(sizeof(struct set_of_paths));
			bl_paths[i]->num_of_paths = 0;
			bl_paths[i]->paths = malloc(NUM_OF_PATHS_PER_QUEUE * sizeof(path*));
			bl_formula[i] = malloc(NUM_OF_PATHS_PER_QUEUE * sizeof(formula*));
			id_paths[i] = NULL;
			id_formula[i] = NULL;
		}
		eqns_full[i] = false;
		eqns_gt1[i] = -1;
		eqns_zero[i] = malloc ((NUM_OF_HEADERS+1) * sizeof(bool));
		block_visited[i] = false;
		idle_visited[i] = NULL;
		for (j=0;j<=NUM_OF_HEADERS;j++)
			eqns_zero[i][j] = false;
	}
	sure = malloc(xMas_network_size*sizeof(bool));
	for (i=0;i<xMas_network_size;i++)
		sure[i] = false;


	float elapsed_init_network = clock()/(float)CLOCKS_PER_SEC;
	printf("Time loading and initializing network: %f secs.\n", elapsed_init_network);

	// Perform simulations to gather typing information on queues or load them from a file.
	printf ("Initializing typing information...\n");
	initialize_types();	

	float elapsed_sims = clock()/(float)CLOCKS_PER_SEC;
	printf("Time loading typing information: %f secs.\n", elapsed_sims-elapsed_init_network);

	for (i=0;i<xMas_network_size&&!found_deadlock;i++) {
		if ((xMas_network[i].type == queue || xMas_network[i].type == buffer) && !sure[i]) {
			bool ret_perm_block = false, ret_perm_idle = false;
			for (flg=0;flg<2&&!found_deadlock ;flg++) {
				// Call the algorithm
				formula* ret;
				if (flg == 0) {
					printf("Determining blocking of queue: %s\n", xMas_network[i].id);
					ret = perm_blocking(i, NUM_OF_HEADERS, 1, 0,0, false);
					ret_perm_block = found_deadlock || check_feasibility_formula(ret, true);
					if (ret_perm_block) {
						printf("Found possible deadlock!\n");
						curr_path->num_of_elts = 0;
					}
					found_deadlock = ret_perm_block;
				}
				else {
					printf("Determining idleness of queue: %s\n", xMas_network[i].id);
					ret = perm_idle(i, NUM_OF_HEADERS, 0, 0, true);
					ret_perm_idle = found_deadlock || check_feasibility_formula(ret, true);
					if (ret_perm_idle) {
						printf("The queue can be idle.\n");
						curr_path->num_of_elts = 0;
					}
					found_deadlock = ret_perm_block && !ret_perm_idle;
				}

				// Free and reinitialize the data structures
				free_formulas(false);
				size_of_garbage = 0;
				for (j=0;j<xMas_network_size;j++) {
					if (bl_paths[j] != NULL) {
						for (k=0;k<bl_paths[j]->num_of_paths;k++)
							path_free(bl_paths[j]->paths[k]);
						bl_paths[j]->num_of_paths = 0;
					}
					if (id_paths[j] != NULL) {
						for (k=0;k<=NUM_OF_HEADERS;k++) {
							for (l=0;l<id_paths[j][k]->num_of_paths;l++)
								path_free(id_paths[j][k]->paths[l]);
							id_paths[j][k]->num_of_paths = 0;
							free(id_paths[j][k]->paths);
							free(id_paths[j][k]);
							free(id_formula[j][k]);
							id_formula[j][k] = NULL;
							id_paths[j][k] = NULL;
						}
						free(id_formula[j]);
						free(id_paths[j]);
						id_paths[j] = NULL;
						id_formula[j] = NULL;
					}
					eqns_full[j] = false;
					eqns_gt1[j] = -1;
					if (!sure[j])
						block_visited[j] = false;
					if (idle_visited[j] != NULL) {
						for (k=0;k<=NUM_OF_HEADERS&&idle_equation_set;k++) {
							idle_visited[j][k] = false;
							eqns_zero[j][k] = false;
						}
						free(idle_visited[j]);
						idle_visited[j] = NULL;
					}
				}
			}

			// Memoize either FALSE or TRUE to the current queue.
			idle_equation_set = false;
			sure[i] = !ret_perm_block && !ret_perm_idle;
			if (!ret_perm_block && !ret_perm_idle)
				printf("Queue %s is live.\n", xMas_network[i].id);
			else if (ret_perm_block)
				printf("Queue is dead (it can be permanently blocked).\n");
			else if (!ret_perm_block && ret_perm_idle)
				printf("Queue cannot be proven live. We have established with certainty that it is not permanently blocked but it might be permanently idle.\n");
			else
				assert(false);
			if (sure[i]) {
				propagate_liveness_forward(xMas_network[i].out[0], xMas_network[i].in_port_connected_to_out[0]);
				propagate_liveness_backward(xMas_network[i].in[0], xMas_network[i].out_port_connected_to_in[0]);
			}
		}
	}
end:
	{}

	float elapsed_total = clock()/(float)CLOCKS_PER_SEC;
	printf("Time algorithm: %f secs.\n", elapsed_total-elapsed_sims);
	printf("Time total: %f\n", elapsed_total);

	for (i=0;i<xMas_network_size;i++) {
		if (eqns_zero[i] != NULL)
			free(eqns_zero[i]);
		if (idle_visited[i] != NULL)
			free(idle_visited[i]);
		if (xMas_network[i].type == queue || xMas_network[i].type == buffer) {
			if (bl_paths[i] != NULL)
				free(bl_paths[i]->paths);
			free(bl_paths[i]);
			free(id_paths[i]);
		}
	}
	free_formulas(true);
	free(block_visited);
	free(idle_visited);
	free(eqns_full);
	free(eqns_gt1);
	free(eqns_zero);
	free(bl_paths);
	free(sure);
	free(garbage);
	free(queue_subject_to_invariants);
	free(type_specific_queues);
	if (false_formula != NULL) free(false_formula);
	if (true_formula != NULL) free(true_formula);
	return 0;
}




