implementation module PMDB

import StdMaybe, StdGeneric
import StdClass, StdBool, StdInt, StdTuple, StdList, StdString, StdFunc, StdArray, StdFile, StdMisc

:: PmdbHistory = History !Int !String !String !PmdbHistory | NoHistory
:: PmdbPath = Pair !PmdbPath !PmdbPath | Edge !String !String

:: PMDB = !{
		file :: !.File,
		mode :: !Int}

:: Chunk a = !{
		write :: !(a *File -> *File),
		read :: !(*File -> *(Bool, a, *File)),
		offset :: !Int,
		value :: !a}

CHAR_SIZE :== 1
INT_SIZE :== IF_INT_64_OR_32 8 4
REAL_SIZE :== 8
NO_OFFSET :== -1

reopen :: !*File !Int -> *File
reopen f m 
	# (ok, f) = freopen f m
	| not ok 
		| m == FReadData = abort "Cannot reopen the file for reading"
		= abort "Cannot reopen the file for writing"
	= f

openPMDB :: !String !*World -> (!Chunk a, !*PMDB, !*World) | pmdb{|*|} a
openPMDB name world 
	# (ok, file, world) = fopen name FAppendData world
	| not ok = abort ("Cannot open the file named `" +++ name +++ "' for writing")
	# (position, file) = fposition file
	| position == 0 = (root default, {mode = FAppendData, file = write` default file}, world)
	# file = reopen file FReadData
	  (ok, existing, file) = read` file
	| not ok = abort ("Cannot read from existing file named `" +++ name +++ "'")
	= (root existing, {mode = FReadData, file = file}, world)
where
	root value = {value = value, offset = 0, write = write`, read = read`}
	read` f = case read f of 
				(Just x, f) -> (True, x, f)
				(_, f) -> (False, default, f)
	write` x f = write size x f
	default = defaultJust maybe
	(maybe, size, write, read, _) = pmdb{|*|} (Edge "" "") NoHistory 0

closePMDB :: !*PMDB !*World -> *World
closePMDB {file} world
	# (ok, world) = fclose file world
	| not ok = abort "Cannot close the file, something whent wrong"
	= world

chunk :: !*PMDB -> (!Chunk a, !*PMDB) | pmdb{|*|} a
chunk db = (defaultJust maybe, db)
where
	(maybe, _, _, _, _) = pmdb{|*|} (Edge "" "") NoHistory 0

chunkValue :: !(Chunk a) !*PMDB -> (!a, !*PMDB)
chunkValue chunk=:{value, read, offset} db
	| offset == NO_OFFSET = (value, db)
	# db=:{file} = openForReading db
	  (ok, file) = fseek file offset FSeekSet
	| not ok = abort "fseek to read failed"
	# (ok, value, file) = read file
	= (value, replaceChunk chunk value offset {db & file = file})
where
	openForReading db=:{file, mode}
		| mode == FReadData = db
		= {db & file = reopen file FReadData, mode = FReadData}

updateChunk :: !(Chunk a) !a !*PMDB -> *PMDB
updateChunk chunk=:{offset, write} value db 
	# db=:{file} = openForWriting db
	  (position, file) = case offset of
	  						NO_OFFSET
	  							# (ok, file) = fseek file 0 FSeekEnd
								| not ok -> abort "fseek to end for writing failed"
								-> fposition file
	  						_ 
								# (ok, file) = fseek file offset FSeekSet
								| not ok -> abort "fseek to offset for writing failed"
	  							-> (offset, file)
	# file = write value file
	  file = replaceChunk chunk value position file
	= {db & file = file} 
where
	openForWriting db=:{file, mode}
		| mode == FAppendData = db
		= {db & file = reopen file FAppendData, mode = FAppendData}

replaceChunk :: !(Chunk a) !a !Int !*env -> *env
replaceChunk chunk value offset db
	# (_, db) = replace (unsafeTypeCast chunk) value offset db
	= db
where
	replace :: !*(Chunk a) !a !Int !*env -> (!*Chunk a, !*env)
	replace c value offset env = ({c & value = value, offset = offset}, env)

	unsafeTypeCast :: !.a -> .b
	unsafeTypeCast _ = code inline {
			pop_a	0
		}

generic pmdb a :: !PmdbPath !PmdbHistory !Int -> (!Maybe a, !Int, !(Int a *File -> *File), !(*File -> *(Maybe a, *File)), !Int)

pmdb{|OBJECT of {gtd_name, gtd_arity, gtd_num_conses}|} pmdb_a path=:(Edge con typ) history i
	| recursion = (Nothing, INT_SIZE, write, read, i) 
	| gtd_num_conses > 255 = abort "Way too many constructors in a single type"
	= (mapMaybe OBJECT a, if pointer INT_SIZE sa, write, read, k)
where
	counts = lookup history
	recursion = counts > gtd_arity
	pointer = counts > 0

	lookup (History n c t hs)
		| c == con && t == typ = n
		| otherwise = lookup hs
	lookup _ = 0

	read file
		| not pointer 
			# (x, file) = ra file
			= (mapMaybe OBJECT x, file)
		# (ok, offset, file) = freadi file
		| not ok = (Nothing, file)
		# (position, file) = fposition file
		  (ok, file) = fseek file offset FSeekSet
		| not ok = abort "fseek to OBJECT failed"
		# (x, file) = ra file
		  (ok, file) = fseek file position FSeekSet
		| not ok = abort "fseek from OBJECT failed"
		= (mapMaybe OBJECT x, file)
			
	write left (OBJECT x) file
		| not pointer = wa sa x file
		# (position, file) = fposition file
		  file = fskip left file
		  (ok, file) = fseek file 0 FSeekEnd
		| not ok = abort "fseek to end for OBJECT failed"
		# (offset, file) = fposition file
		  file = wa sa x file
		  (ok, file) = fseek file position FSeekSet
		| not ok = abort "fseek back to OBJECT failed"
		= fwritei offset file

	(a, sa, wa, ra, k) = pmdb_a path history i

pmdb{|OBJECT|} pmdb_a path history i = (mapMaybe OBJECT a, s, \l (OBJECT x) f -> w l x f, \f -> case r f of (x, f) -> (mapMaybe OBJECT x, f), j)
where
	(a, s, w, r, j) = pmdb_a path history i

pmdb{|EITHER|} pmdb_a pmdb_b path history i 
	= (either, se, write, read, k)
where
	se = max sa sb

	(a, sa, wa, ra, j) = pmdb_a path history i
	(b, sb, wb, rb, k) = pmdb_b path history j

	read file 
		# (position, file) = fposition file
		  (x, file) = ra file
		= case x of
			Just x -> (Just (LEFT x), skip position file)
			_
				# (ok, file) = fseek file position FSeekSet
				| not ok -> abort "fseek retry CONS failed"
				# (y, file) = rb file
				-> case y of
					Just y -> (Just (RIGHT y), skip position file)
					_	-> (Nothing, file)
	where
		skip position file
			# (ok, file) = fseek file (position + se) FSeekSet
			| not ok = abort "fseek after CONS failed"
			= file

	write left (LEFT x) file = wa left x file
	write left (RIGHT y) file = wb left y file
	
	either
		| sa <= sb = case a of Just x -> Just (LEFT x); _ -> mapMaybe RIGHT b
		= case b of Just y -> Just (RIGHT y); _ -> mapMaybe LEFT a

pmdb{|CONS of {gcd_name, gcd_arity, gcd_type, gcd_index, gcd_type_def={gtd_name, gtd_arity, gtd_num_conses}}|} pmdb_a path history i
	= (mapMaybe CONS a, if single sa (sa + CHAR_SIZE), write, read, k)
where
	single = gtd_num_conses < 2

	read file
		| single
			# (x, file) = ra file
			= (mapMaybe CONS x, file)
		# (ok, index, file) = freadc file
		| not ok || toInt index <> gcd_index = (Nothing, file)
		# (x, file) = ra file
		= (mapMaybe CONS x, file) 
	
	write left (CONS x) file
		| single 
			# file = wa left x file
			= fskip (left - sa) file
		# file = fwritec (toChar gcd_index) file
		  left = left - CHAR_SIZE
		  file = wa left x file
		= fskip (left - sa) file
	
	(a, sa, wa, ra, k) = pmdb_a path` history` j

	(path`, _) = makePairs gcd_arity formals
	where
		makePairs :: !Int ![GenType] -> (!PmdbPath, ![GenType])
		makePairs 0 fs = (Edge gcd_name gtd_name, fs)
		makePairs 1 [f:fs] = (Edge gcd_name (if isvar gtd_name typecons), fs)
		where
			(isvar, typecons) = typeCons f False
		makePairs n fs 
			# (a, fs) = makePairs (n >> 1) fs
			  (b, fs) = makePairs ((n + 1) >> 1) fs
			= (Pair a b, fs) 

	typeCons :: !GenType !Bool -> (!Bool, !String)
	typeCons (GenTypeApp x y) _ = typeCons x True
	typeCons (GenTypeCons x) _ = (False, x)
	typeCons (GenTypeVar x) isvar = (isvar, "")
	typeCons (GenTypeArrow x y) _ = (False, "(" +++ snd (typeCons x True) +++ " -> " +++ snd (typeCons y True) +++ ")")
	
	history` = foldl increment history (typeConses gcd_type [])
	where
		typeConses :: !GenType ![(String, String)] -> [(String, String)]
		typeConses (GenTypeArrow arg res) acc = typeConses res (if isvar acc [(gcd_name, tc):acc])
		where
			(isvar, tc) = typeCons arg True
		typeConses _ acc = acc

		increment (History n c t hs) (con, typ)
			| c == con && t == typ = History (n + 1) c t hs
			| otherwise = History n c t (increment hs (con, typ))
		increment _ (con, typ) = History 1 con typ NoHistory
		
	(formals, result) = splitType gcd_arity t
	where
		splitType :: !Int !GenType -> (![GenType], !GenType)
		splitType 0 t = ([], t)
		splitType n (GenTypeArrow x y) = ([x:xs], r)
			where (xs, r) = splitType (n - 1) y
			
	(t, j) = (gcd_type, i) //freshCopy gcd_type i

pmdb{|CONS|} pmdb_a path history i = (mapMaybe CONS a, s, \l (CONS x) f -> w l x f, \f -> case r f of (x, f) -> (mapMaybe CONS x, f), j)
where
	(a, s, w, r, j) = pmdb_a path history i

pmdb{|PAIR|} pmdb_a pmdb_b path history i
	= (pair, sa + sb, write, read, k)
where
	(a, sa, wa, ra, j) = pmdb_a pa history i
	(b, sb, wb, rb, k) = pmdb_b pb history j

	(pa, pb) = case path of 
					Pair l r -> (l, r)
					_ -> (path, path)

	read file
		# (x, file) = ra file
		  (y, file) = rb file
		= case (x, y) of
			(Just x, Just y) -> (Just (PAIR x y), file)
			_ -> (Nothing, file)

	write left (PAIR x y) file
		# file = wa left x file
		= wb (left - sa) y file

	pair = case (a, b) of
				(Just x, Just y) -> Just (PAIR x y)
				_ -> Nothing
			
pmdb{|FIELD|} pmdb_a path history i = (mapMaybe FIELD a, s, \l (FIELD x) f = w l x f, \f -> case r f of (x, f) -> (mapMaybe FIELD x, f), j)
where
	(a, s, w, r, j) = pmdb_a path history i

pmdb{|UNIT|} path history i = (Just UNIT, 0, write, read, i)
where
	read file = (Just UNIT, file)
	write _ _ file = file

pmdb{|Int|} path history i = (Just 0, INT_SIZE, write, read, i)
where
	read file
		# (ok, x, file) = freadi file
		| not ok = (Nothing, file)
		= (Just x, file)

	write _ x file = fwritei x file

pmdb{|Char|} path history i = (Just '\0', CHAR_SIZE, write, read, i)
where
	read file
		# (ok, x, file) = freadc file
		| not ok = (Nothing, file)
		= (Just x, file)

	write _ x file = fwritec x file

pmdb{|Bool|} path history i = (Just False, CHAR_SIZE, write, read, i)
where
	read file
		# (ok, x, file) = freadc file
		| not ok = (Nothing, file)
		= (Just (x <> '\0'), file)

	write _ x file = fwritec (if x '\255' '\0') file

pmdb{|Real|} path history i = (Just 0.0, REAL_SIZE, write, read, i)
where
	read file
		# (ok, x, file) = freadr file
		| not ok = (Nothing, file)
		= (Just x, file)

	write _ x file = fwriter x file

pmdb{|{!}|} pmdb_a path history i = (Just {}, ARRAY_SIZE, writeArray s w, readArray a s r, j)
where
	(a, s, w, r, j) = pmdb_a path history i

pmdb{|{}|} pmdb_a path history i = (Just {}, ARRAY_SIZE, writeArray s w, readArray a s r, j)
where
	(a, s, w, r, j) = pmdb_a path history i
	
pmdb{|String|} path history i = (Just {}, ARRAY_SIZE, writeArray s w, readArray a s r, j)
where
	(a, s, w, r, j) = pmdb{|*|} path history i

pmdb{|Chunk|} pmdb_a path history i = (c, INT_SIZE, write, read, i)
where
	c = mapMaybe (chunkAt NO_OFFSET) a

	chunkAt offset value = {value = value, offset = offset, write = wa sa, read = refresh}
	
	d = defaultJust a
	
	read file 
		# (ok, offset, file) = freadi file
		| not ok = abort "fread offset of Chunk failed"
		= (Just (chunkAt offset d), file)
	
	write left chunk=:{value, offset, write} file 
		| offset <> NO_OFFSET = fwritei offset file
		# (position, file) = fposition file
		  file = fskip left file
		  (ok, file) = fseek file 0 FSeekEnd
		| not ok = abort "fseek to Chunk failed"
		# (offset, file) = fposition file
		  file = replaceChunk chunk value offset file 
		  file = write value file
		  (ok, file) = fseek file position FSeekSet
		| not ok = abort "fseek from Chunk failed"
		= fwritei offset file

	refresh file
		# (maybe, file) = ra file
		= case maybe of
			Just x -> (True, x, file)
			_ -> (False, d, file)
	
	(a, sa, wa, ra, k) = pmdb_a path history i

ARRAY_SIZE :== INT_SIZE * 3

readArray :: !(Maybe e) !Int !(*File -> *(Maybe e, *File)) !*File -> (!Maybe (a e), !*File) | Array a e
readArray a sa ra file
	# (ok, reserved, file) = freadi file
	| not ok = (Nothing, file)
	| reserved <= 0 = unpack (~ reserved) file
	# (ok, offset, file) = freadi file
	| not ok = abort "fread offset of array failed"
	# (ok, elements, file) = freadi file
	| not ok = abort "fread size of array failed"
	# (position, file) = fposition file
	  (ok, file) = fseek file offset FSeekSet
	| not ok = abort "fseek to array failed"
	# (maybe, file) = freadarray (createArray elements e) 0 file
	  (ok, file) = fseek file position FSeekSet
	| not ok = abort "fseek from array failed"
	= (maybe, file)
where
	unpack elements file
		# (maybe, file) = freadarray (createArray elements e) 0 file
		  (ok, file) = fseek file (ARRAY_SIZE - INT_SIZE - elements * sa) FSeekCur
		| not ok = abort "fseek over array failed"
		= (maybe, file)

	e = defaultJust a

	freadarray array index file 
		| index >= size array = (Just array, file)
		# (maybe, file) = ra file
		= case maybe of
			Just x -> freadarray {array & [index] = x} (index + 1) file
			_ -> (Nothing, file)

writeArray :: !Int !(Int e -> (*File -> *File)) !Int !(a e) !*File -> *File | Array a e
writeArray sa wa left array file
	# (position, file) = fposition file
	  file = fskip left file
	  file = reopen file FReadData
	  (ok, file) = fseek file position FSeekSet
	| not ok = abort "fseek to array size failed"
	# (ok, reserved, file) = freadi file
	  reserved = if ok reserved -1
	| elements <= reserved = update position reserved file
	| inlined <= ARRAY_SIZE = pack position file
	# file = reopen file FAppendData
	  (offset, file) = fposition file
	= store position reserved offset file
where
	update position reserved file
		# (ok, offset, file) = freadi file
		| not ok = abort "fread of array offset failed"
		# file = reopen file FAppendData
		  (ok, file) = fseek file offset FSeekSet
		| not ok = abort "fseek to update array failed"
		= store position reserved offset file
	
	pack position file
		# file = reopen file FAppendData
		  (ok, file) = fseek file position FSeekSet
		| not ok = abort "fseek back to array failed"
		# file = fwritei (~ elements) file
		  file = fwritearray 0 (ARRAY_SIZE - INT_SIZE) array file
		= fskip (ARRAY_SIZE - inlined) file

	store position reserved offset file
		# file = fwritearray 0 totalsize array file 
		  (ok, file) = fseek file position FSeekSet
		| not ok = abort "fseek from array failed"
		# file = fwritei (max reserved elements) file
		  file = fwritei offset file
		= fwritei elements file
	
	elements = size array
	
	totalsize = elements * sa
	
	inlined = INT_SIZE + totalsize

	fwritearray index left array file
		| index >= size array = file
		# file = wa left array.[index] file
		= fwritearray (index + 1) (left - sa) array file 

defaultJust :: !(Maybe a) -> a
defaultJust (Just x) = x
defaultJust _ = abort "Construction of a default value failed"

fskip :: !Int !*File -> *File
fskip n file
	| n <= 0 = file
	# (position, file) = fposition file
	  (ok, file) = fseek file 0 FSeekEnd
	| not ok = abort "fseek to check end of file failed"
	# (end, file) = fposition file
	  diff = end - position
	| n > diff = fappend (n - diff) fskipbuffer file 
	# (ok, file) = fseek file (position + n) FSeekSet
	| not ok = abort "fseek while skipping failed"
	= file
where
	fappend :: !Int !String !*File -> *File
	fappend n s file
		| n <= size s = fwritesubstring 0 n s file
		= fappend (n - size s) s (fwrites s file)

fskipbuffer :: String
fskipbuffer =: createArray (256 * INT_SIZE) '\0'
/*
unify :: ![GenType] ![GenType] !GenType -> GenType
unify [GenTypeVar x:xs] [GenTypeVar y:ys] t | x == y = unify xs ys t
unify [GenTypeVar x:xs] [y:ys] t
/*	| not (occurs x y)*/ = unify (map s xs) (map s ys) (s t) where s = subst x y
unify [x:xs] [GenTypeVar y:ys] t
/*	| not (occurs y x)*/ = unify (map s xs) (map s ys) (s t) where s = subst y x
unify [GenTypeApp x1 x2:xs] [GenTypeApp y1 y2:ys] t	= unify [x1, x2:xs] [y1, y2:ys] t
unify [GenTypeArrow x1 x2:xs] [GenTypeArrow y1 y2:ys] t = unify [x1, x2:xs] [y1, y2:ys] t
unify [GenTypeCons x:xs] [GenTypeCons y:ys] t | x == y = unify xs ys t
unify [] [] t = t
/*unify xs ys = abort ("Cannot unify " +++ separatorList "," xs +++ " and " +++ separatorList "," ys)

separatorList :: !String ![a] -> String | toString a
separatorList s xs = f xs
where
	f [x] = toString x
	f [x:xs] = toString x +++ s +++ f xs
	f _ = ""

occurs :: !String !GenType -> Bool
occurs v (GenTypeVar x) = v == x
occurs v (GenTypeCons x) = v == x
occurs v (GenTypeApp x y) = occurs v x || occurs v y
occurs v (GenTypeArrow x y) = occurs v x || occurs v y
*/
subst :: !String !GenType !GenType -> GenType
subst x y t = s t
where
	s (GenTypeApp a b) = GenTypeApp (s a) (s b)
	s (GenTypeArrow a b) = GenTypeArrow (s a) (s b)
	s (GenTypeVar t) | t == x = y
	s t = t

freshCopy :: !GenType !Int -> (!GenType, !Int)
freshCopy t fresh = (s t, fresh`)
where
	(s, fresh`) = makeSubst t fresh
	
	makeSubst :: !GenType !Int -> (!(GenType -> GenType), !Int)
	makeSubst (GenTypeVar x) fresh = (subst x (GenTypeVar (toString fresh)), fresh + 1)
	makeSubst (GenTypeApp x y) fresh = (s2 o s1, fresh``)
	where 
		(s1, fresh`) = makeSubst x fresh
		(s2, fresh``) = makeSubst y fresh`
	makeSubst (GenTypeArrow x y) fresh = (s2 o s1, fresh``)
	where 
		(s1, fresh`) = makeSubst x fresh
		(s2, fresh``) = makeSubst y fresh`
	makeSubst _ fresh = (id, fresh)

instance toString GenType where
	toString x = f x False
	where
		f (GenTypeVar x) _ = x
		f (GenTypeCons x) _ = x
		f (GenTypeArrow x y) False = f x True +++ " -> " +++ f y False
		f (GenTypeApp x y) False = f x False +++ " " +++ f y True
		f t True = "(" +++ f t False +++ ")"
*/