module Emv

open Crypto
open Data
open SysLib
open Pi

// Issuer public/private key pair 
let sI = rsa_keygen ()
let pI = rsa_pub sI

// Issuer Master Key
let mkI = hmac_keygen ()

// ICC Application Cryptogram Master Key
let create_mkAC pan = (mkI, pan)

// ICC Application Cryptogram Session Key
let create_skAC mkAC atc = (mkAC, atc)

// Events used in queries
type event =
| CardTransactionInit of bytes * bool * bool * bool * bytes
| CardTransactionFinish of bool * bool * bool * bytes * bytes * bool
| TerminalTransactionFinish of bool * bool * bool * bytes * bytes * bool
| CardVerifyPIN of bool
| TerminalVerifyPIN of bool
| TerminalSDA of bool * bytes
| TerminalDDA of bool * bytes
| TerminalCDA of bool * bytes * bytes
| TerminalCDA2 of bool * bytes * bytes
| Nothing // Empty event

let tr: event Pi.trace = Pi.trace()

// Create the SSAD element for the card
let create_ssad d =
  rsa_sign sI (sha1 d)

// Create a certificate for data using the key sk 
let create_cert sk data =
  rsa_encrypt sk data

// Construct CDA signature on message, if requested
let cda_sig sIC data cda_requested cda_enabled =
  // If CDA is enabled, add a signature over the data in the AC
  if cda_enabled then
    begin
    if cda_requested then
      rsa_sign sIC data
    else
      0
    end
  else
    failwith "CDA not supported by card"

// Construct the MAC used in the response to GENERATE_AC commands
let construct_ac_mac skAC data =
  hmacsha1 skAC data

// Perform the actual transaction.
//   msg is the last unprossed incoming APDU
let card_transaction (c,atc,(sIC,pIC),skAC,nonceC) msg pdol aip pan force_online =
  let (sda_enabled,dda_enabled,cda_enabled) = aip in
  let (ac_type, cda_requested, cdol1) = APDU.parse_generate_ac msg in
  let (amount, cvr, nonceT) = cdol1 in
  let mac = construct_ac_mac skAC (amount, nonceT, atc) in
  if ac_type = Data.ARQC then
    begin
    Net.send c (APDU.generate_ac_response Data.ARQC atc mac (cda_sig sIC (Data.ARQC, atc, pdol, cdol1, mac) cda_requested cda_enabled));

    let (ac_type, cda_requested, cdol2) = APDU.parse_generate_ac (Net.recv c) in
    if ac_type = Data.TC then
      begin
      log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,true));
      Net.send c (APDU.generate_ac_response Data.TC atc mac (cda_sig sIC (Data.TC, atc, pdol, cdol1, cdol2, mac) cda_requested cda_enabled))
      end
    elif ac_type = Data.AAC then
      begin
      log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false));
      Net.send c (APDU.generate_ac_response Data.AAC atc mac 0)
      end
    else
      failwith "Unsupported command"
    end
  elif ac_type = Data.TC then
    begin
    if force_online then
      begin
      Net.send c (APDU.generate_ac_response Data.ARQC atc mac (cda_sig sIC (Data.ARQC, atc, pdol, cdol1, mac) cda_requested cda_enabled));
      let (ac_type, cda_requested, cdol2) = APDU.parse_generate_ac (Net.recv c) in

      if ac_type = Data.TC then
        begin
        log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,true));
        Net.send c (APDU.generate_ac_response Data.TC atc mac (cda_sig sIC (Data.TC, atc, pdol, cdol1, cdol2, mac) cda_requested cda_enabled))
        end
      elif ac_type = Data.AAC then
        begin
        log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false));
        Net.send c (APDU.generate_ac_response Data.AAC atc mac 0)
        end
      else
        failwith "Unsupported command"
      end
    else
      begin
      log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,true));
      Net.send c (APDU.generate_ac_response Data.TC atc mac (cda_sig sIC (Data.TC, atc, pdol, cdol1, mac) cda_requested cda_enabled))
      end
    end
  elif ac_type = Data.AAC then
    begin
    log tr (CardTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false));
    Net.send c (APDU.generate_ac_response Data.AAC atc mac 0)
    end
  else
    failwith "Unsupported command"

// Perform PIN verification if requested, otherwise do nothing
//   msg is the last unprocessed incoming APDU
//   this function returns the last unprocessed incoming APDU
let card_pin_verify (c,atc,(sIC,pIC)) msg =
  // Customer verification
  if Data.VERIFY = APDU.get_command msg then
    begin
    let pin = APDU.parse_verify msg in
    if pin = "1234" then
      begin
      log tr (CardVerifyPIN(true));
      Net.send c (APDU.verify_response true)
      end
    else
      Net.send c (APDU.verify_response false);
    Net.recv c
    end
  else
    msg
 
// Perform DDA Authentication if requested, otherwise do nothing
//   this function returns the last unprocessed incoming APDU
let card_dda (c,atc,(sIC,pIC),nonceC) dda_enabled = 
  let msg = Net.recv c in
  if Data.INTERNAL_AUTHENTICATE = APDU.get_command msg then
    if dda_enabled then
      begin
      let nonceT = APDU.parse_internal_authenticate msg in
      let signature = rsa_sign sIC (nonceC, nonceT) in
      Net.send c (APDU.internal_authenticate_response nonceC signature);
      Net.recv c
      end
    else
      failwith "DDA not supported by card"
  else
    msg

// Process performing a single transaction for a card
let card_process (sIC, pIC, mkAC, pan) c (sda_enabled, dda_enabled, cda_enabled) = 
  let force_online = Net.recv c in

  // Construct the AIP
  let aip = (sda_enabled, dda_enabled, cda_enabled) in
  // Construct the AFL
  let afl = "" in

  // SELECT APPLICATION command
  APDU.parse_select_application (Net.recv c);

  // Send response with empty PDOL
  Net.send c APDU.select_application_response;

  // Construct Application Transaction Counter
  let atc = mkNonce () in
  let nonceC = mkNonceN 8 in

  // Create session key
  let skAC = create_skAC mkAC atc in

  // Generate event for initialization of transaction
  log tr (CardTransactionInit(atc,sda_enabled,dda_enabled,cda_enabled,pan));

  // GET PROCESSING OPTIONS command
  let pdol = APDU.parse_get_processing_options (Net.recv c) in
  // Send response with AIP and AFL
  Net.send c (APDU.get_processing_options_response aip afl);

  // READ RECORD command
  APDU.parse_read_record (Net.recv c);
  // Send response
  Net.send c (APDU.read_record_response ((create_ssad (aip, pan)), (create_cert sI (pIC, (sha1 (aip, pan)))), pan));

  // Perform DDA if enabled
  let msg = card_dda (c,atc,(sIC,pIC),nonceC) dda_enabled in

  // Perform PIN verification if requested
  let msg = card_pin_verify (c,atc,(sIC,pIC)) msg in

  // Perform the actual transaction
  card_transaction (c,atc,(sIC,pIC),skAC,nonceC) msg pdol aip pan force_online

// Main process for the card
let card () =
  // Set up channel between card and terminal
  let c = Net.listen "" in

  // Initialise card dependent data
  let sIC = rsa_keygen () in
  let pIC = rsa_pub sIC in
  let pan = mkNonce () in
  let mkAC = create_mkAC pan in

  let (sda_enabled, dda_enabled, cda_enabled) = Net.recv c in

  Pi.fork (fun() -> card_process (sIC, pIC, mkAC, pan) c (sda_enabled, dda_enabled, cda_enabled))

// Main process for the terminal
let terminal () = 
  // Set up channel between card and terminal
  let c = Net.connect "" in

  // Get random transaction & terminal options from network
  let (offline_pin_enabled, online_enabled, amount) = Net.recv c in

  // Create TVR
  let tvr = mkDb "tvr" in
  // Create CVR
  let cvr = mkDb "cvr" in

  // Initialize transaction dependent values
  let terminal_country_code = "0528" in // Netherlands
  let transaction_currency_code = "0978" in // Euro

  // Select application
  Net.send c APDU.select_application;

  // Receive PDOL
  let pdol = APDU.parse_select_application_response (Net.recv c) in
  let pdol_items = () in

  // Get processing options
  Net.send c (APDU.get_processing_options pdol_items);
  let (aip, afl) = APDU.parse_get_processing_options_response (Net.recv c) in
  let (sda_enabled, dda_enabled, cda_enabled) = aip in

  // Read files  
  Net.send c APDU.read_record;
  let (ssad, cert, pan) = APDU.parse_read_record_response (Net.recv c) in

  // Perform SDA authentication if this is the highest supported authentication method
  if cda_enabled = false then
    if dda_enabled = false then
      if sda_enabled then
        begin
        let result_sda = rsa_verify_no_fail pI (sha1 (aip, pan)) ssad in
        log tr (TerminalSDA(result_sda,pan))
        end
      else
        log tr (Nothing)
    else
      log tr (Nothing)
  else
    log tr (Nothing);

  // Perform DDA authentication if this is the highest supported authentication method
  if cda_enabled = false then
    if dda_enabled then
      begin
      let (pIC, sda_sha) = rsa_decrypt pI cert in
      if sda_sha = (sha1 (aip,pan)) then
        begin
        let nonceT = (mkNonceN 4) in
        Net.send c (APDU.internal_authenticate nonceT);
        let (nonceC, signature) = APDU.parse_internal_authenticate_response (Net.recv c) in
        let result_dda = rsa_verify_no_fail pIC (nonceC, nonceT) signature in
        log tr (TerminalDDA(result_dda, pan))
        end
      else
        log tr (TerminalDDA(false, pan))
      end
    else
      log tr (Nothing)
  else
    log tr (Nothing);

  // Perform offline plaintext PIN verification
  if offline_pin_enabled then
    begin
    let pin = "1235" in
    Net.send c (APDU.verify pin);
    let response = APDU.parse_verify_response (Net.recv c) in
    log tr (TerminalVerifyPIN(response))
    end
  else
    log tr (Nothing);

  // Perform the actual transaction
  let nonceT = mkNonceN 4 in
  let cdol1 = (amount, cvr, nonceT) in
  let cdol2 = (nonceT) in

  if online_enabled then
    Net.send c (APDU.generate_ac Data.ARQC cda_enabled cdol1)
  else
    Net.send c (APDU.generate_ac Data.TC cda_enabled cdol1);

  let (ac_type, atc, ac, signature) = APDU.parse_generate_ac_response (Net.recv c) in

  // CDA is performed if this is supported
  if cda_enabled = true then
    begin
    let (pIC, sda_sha) = rsa_decrypt pI cert in
    if sda_sha = sha1((aip,pan)) then
      let result_cda = rsa_verify_no_fail pIC (ac_type, atc, pdol_items, cdol1, ac) signature in
      log tr (TerminalCDA(result_cda,atc,ac_type));

      if result_cda = false then
        begin
        log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false));
        failwith "CDA failed"
        end
      else
        log tr (Nothing)
(* Following code leads to very long execution time! Why?
      if result_cda then
        log tr (TerminalCDA(true,atc,ac_type))
      else
        begin
        log tr (TerminalCDA(false,atc,ac_type));
        failwith "CDA failed"
        end
*)
    else
      begin
      log tr (TerminalCDA(false,atc,ac_type));
      failwith "CDA failed"
      end
    end
  else
    log tr (Nothing);

  if ac_type = Data.ARQC then
    begin
    Net.send c (APDU.generate_ac Data.TC cda_enabled cdol2);
    let (ac_type2, atc2, ac2, signature2) = APDU.parse_generate_ac_response (Net.recv c) in
    if atc = atc2 then
      begin
      if ac_type2 = Data.TC then
        begin
        if cda_enabled = true then
          begin
          let (pIC, sda_sha) = rsa_decrypt pI cert in
          if sda_sha = sha1((aip,pan)) then
            begin
            let result_cda2 = rsa_verify_no_fail pIC (ac_type2, atc2, pdol_items, cdol1, cdol2, ac2) signature2 in
            log tr (TerminalCDA2(result_cda2,atc2,ac_type2));
            log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,result_cda2))
            end
          else
            begin
            log tr (TerminalCDA2(false,atc2,ac_type2));
            log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false))
            end
          end
        else
          log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,true))
        end
      elif ac_type = Data.AAC then
        // Abort transaction
        log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false))
      else
        failwith "Unexpected AC type"
      end
    else
      failwith "Unexpected change of ATC"
    end
  elif ac_type = Data.AAC then
    // Abort transaction
    log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,false))
  elif online_enabled = false then
    // AC type is TC and online is not enabled
    log tr (TerminalTransactionFinish(sda_enabled,dda_enabled,cda_enabled,pan,atc,true))
  else
    failwith "Unexpected AC type"
