module Crypto

open Data

type key                            (* Cryptographic keys *)

val mkNonce: unit -> bytes (* Generate a fresh nonce (16 bytes) *)
val sha1: bytes -> bytes   (* Compute the SHA-1 hash of a byte array *)

val hmac_keygen: unit -> key        (* Generate a pseudorandom HMAC key *)
val hmacsha1: key -> bytes -> bytes (* Create a MAC with a key and a message *)
val hmacsha1_verify: key -> bytes -> bytes -> unit (* Verify MAC *)

val aes_keygen:  unit -> key           (* Generate a pseudorandom AES key *)
val aes_encrypt: key -> bytes -> bytes (* Encrypt a message with a key *)
val aes_decrypt: key -> bytes -> bytes (* Decrypt a message with a key *)

val symkeyToBytes: key -> bytes     (* Extract underlying bytes from a key *)
val symkey: bytes -> key            (* Repackage extracted bytes as key *)

val rsa_keygen: unit -> key            (* Generate a RSA key-pair *)
val rsa_pub: key -> key                (* Extract the public key *)

val rsaKeyToBytes: key -> bytes    (* Extract underlying bytes from RSA key *)
val rsaKeyFromBytes: bytes -> key  (* Repackage extracted bytes as RSA key *)
 
val rsa_encrypt: key -> bytes -> bytes (* Encrypt using an RSA public key *)
val rsa_decrypt: key -> bytes -> bytes (* Decrypt using an RSA private key *)
  
val rsa_sign: key -> bytes -> bytes    (* Sign using an RSA private key *)
val rsa_verify: key -> bytes -> bytes -> unit   (* Verify signature *)

val rsa_verify_no_fail: key -> bytes -> bytes -> bool   (* Verify signature and return result *)
val mkNonceN: int -> bytes (* Generate a fresh nonce (N bytes) *)

