module Data

open SysLib

type command =
| SELECT_APPLICATION
| GET_PROCESSING_OPTIONS
| READ_RECORD
| INTERNAL_AUTHENTICATE
| VERIFY
| GENERATE_AC

type ac_type =
| ARQC
| AAC
| TC

type str =
  | Literal of string          (* Literal strings, e.g. "abc" *)
  | Base64 of bytes            (* Base64 encoded bytes *)
and  bytes =
  | Random of name             (* Randomly generated bytestring *)
  | Concat of bytes * bytes    (* Concatenation of bytestrings *)
  | Concat3 of bytes * bytes * bytes   (* Concatenation of bytestrings *)
  | Concat4 of bytes * bytes * bytes * bytes  (* Concatenation of bytestrings *)
  | Utf8 of str                (* Utf8 encoded string *)
  | Hash of bytes              (* Hash of a bytes *)
  | MAC of key * bytes         (* MAC using a key and a bytes *)
  | SymEncrypt of key * bytes  (* Symmetric Encryption *)
  | Sign of key * bytes        (* Public-key signature *)
  | AsymEncrypt of key * bytes (* Asymmetric Encryption *)
  | Boolean of bool            (* Encoded boolean *)
  | Fresh of name
  | AcType of ac_type
  | Command of command
and name = Pi.name

and key =
  | SymKey of name             (* Symmetric key for MAC, SymEncrypt *)
  | PrivateKey of name         (* Private key for Sign, AsymDecrypt *)
  | PublicKey of name          (* Public key corresponding to private key *)

let str s = Literal s
let istr ss = 
  match ss with
      Literal s -> s
    | _ -> failwith "Cannot extract string from non-literal"

let bytes (b:byte[]) : bytes = 0 // Not implemented
let ibytes (b:bytes) : byte[] =  0 //Not implemented

let utf8 s = Utf8 s
let iutf8 b = 
  match b with 
      Utf8(s) -> s 
    | _ -> failwith "Attempt to decode non-Utf8 encoded bytestring"

let base64 b = Base64 b
let ibase64 s = 
  match s with 
      Base64(b) -> b
    | _ -> failwith "Attempt to decode non-Base64 encoded string"

let concat b1 b2 = Concat(b1,b2)
let iconcat b = 
  match b with 
      Concat(b1,b2) -> (b1,b2)
    | _ -> failwith "Attempt to iconcat non-concatenated bytestring"

let freshbytes s = Pi.name s
let random () = Random(freshbytes "random")
let mkSymKey () = SymKey(freshbytes "symkey")
let mkPrivateKey () = PrivateKey(freshbytes "privkey")

let publicKey k = 
  match k with
      PrivateKey(k) -> PublicKey(k)
    | _ -> failwith "Cannot extract public key from non-private key"

let symTag = "SymKey"
let privTag = "PrivateKey"
let pubTag = "PublicKey"

let keyToBytes k = 
  match k with
      SymKey(n) -> Concat(symTag,Random(n))
    | PrivateKey(n) -> Concat(privTag,Random(n))
    | PublicKey(n) -> Concat(pubTag,Random(n))

let keyFromBytes b = 
  match b with
    | Concat(t,Random(n)) when t = symTag -> SymKey(n)
    | Concat(t,Random(n)) when t = privTag -> PrivateKey(n)
    | Concat(t,Random(n)) when t = pubTag -> PublicKey(n)
    | _ -> failwith "Cannot extract key from non-serialized keybytes"

let hash b = Hash(b)
let mac k b = MAC(k,b)

let sym_encrypt k b = SymEncrypt(k,b)
let sym_decrypt k e = 
  match e with
      SymEncrypt(kk,bb) when k = kk -> bb
    | _ -> failwith "Symmetric decryption failed"

let sign k b = Sign(k,b)
let verify k b sv = 
  match sv with
      Sign(PrivateKey(kk),bb) when PublicKey(kk) = k -> if bb = b then () else failwith "verify: signed values do not match"
    | _ -> failwith "Signature verification failed"

let asym_encrypt k b = AsymEncrypt(k,b)
let asym_decrypt k e = 
  match e with
      AsymEncrypt(PublicKey(kk),bb) when k = PrivateKey(kk) -> bb
    | AsymEncrypt(PrivateKey(kk),bb) when k = PublicKey(kk) -> bb
    | _ -> failwith "Asymmetric decryption failed"
  
let equals (b1:bytes) (b2:bytes) = if b1 = b2 then true else false

let fpstr (o: out_channel) (s: str): unit = ()
let fpbytes (o: out_channel) (b: bytes): unit = ()

let syms_str s = ()
let syms_bytes b = ()

let size_str (s: str): int = syms_str s
let size_bytes (b: bytes): int = syms_bytes b

let verify_no_fail k b sv = 
  match sv with
  | Sign(PrivateKey(kk),bb) when PublicKey(kk) = k -> if bb = b then true else false
  | _ -> false

let not b =
  if b then false
  else true
