// This file is part of the Java Card Firewall Tester program.
// 
// Authors: 
// 
// Wojciech Mostowski, woj@cs.ru.nl
// Erik Poll, erikpoll@cs.ru.nl
// Radboud University Nijmegen
// The Netherlands
// 
// Copyright (c) Wojciech Mostowski, Erik Poll,
// Radboud University Nijmegen (RU),
// Stichting Technische Wetenschappen (STW)
// 
// The Java Card Firewall Tester has been developed for the PinPas Java
// Card project, see http://www.win.tue.nl/pinpasjc/. The program is
// distributed under the licence terms that can be found in the LICENCE
// file in the main installation directory. Please refer to the LICENCE &
// README files for further information.

package firewallinterface;

import javacard.framework.*;

import javacardx.crypto.*;

import javacard.security.*;

public class ShareableObject implements TestInterface1 {

    private byte[] arrayPersistent = null;
    private byte[] arrayReset = null;
    private byte[] arrayDeselect = null;
    private byte[] tmp;
    private Cipher intCipher, extCipher;


    public ShareableObject(short size) {
        arrayPersistent = new byte[size];
        arrayReset      = JCSystem.makeTransientByteArray(size, JCSystem.CLEAR_ON_RESET);
        arrayDeselect   = JCSystem.makeTransientByteArray(size, JCSystem.CLEAR_ON_DESELECT);
        intCipher = Cipher.getInstance(Cipher.ALG_DES_ECB_NOPAD, false);
        extCipher = Cipher.getInstance(Cipher.ALG_DES_ECB_NOPAD, true);
    }

    public byte[] getArray(byte type) {
        switch (type) {
        case JCSystem.NOT_A_TRANSIENT_OBJECT:
            return arrayPersistent;
        case JCSystem.CLEAR_ON_RESET:
            return arrayReset;
        case JCSystem.CLEAR_ON_DESELECT:
            return arrayDeselect;
        default:
            return null;
        }
    }

    public byte[] getArray(byte type, byte value) {
        byte[] result = null;
        switch (type) {
        case JCSystem.NOT_A_TRANSIENT_OBJECT:
            result = arrayPersistent;
            break;
        case JCSystem.CLEAR_ON_RESET:
            result = arrayReset;
            break;
        case JCSystem.CLEAR_ON_DESELECT:
            result = arrayDeselect;
            break;
        default:
            result = null;
        }
        Util.arrayFillNonAtomic(result, ZERO, (short)result.length, value);
        return result;
    }

    public boolean checkAID(byte[] aid, byte length) {
        return JCSystem.getAID().equals(aid, ZERO, length);
    }

    public boolean checkPreviousAID(byte[] aid, byte length) {
        return JCSystem.getPreviousContextAID().equals(aid, ZERO, length);
    }

    public void createArray(short size, byte type) {
        if (type == JCSystem.NOT_A_TRANSIENT_OBJECT)
            tmp = new byte[size];
        else
            tmp = JCSystem.makeTransientByteArray(size, type);
    }

    public void readWriteArray(byte type) {
        switch (type) {
        case JCSystem.NOT_A_TRANSIENT_OBJECT:
            tmp = arrayPersistent;
            break;
        case JCSystem.CLEAR_ON_RESET:
            tmp = arrayReset;
            break;
        case JCSystem.CLEAR_ON_DESELECT:
            tmp = arrayDeselect;
            break;
        default:
            tmp = null;
        }
        byte b = tmp[0];
        tmp[1] = 0x01;
    }

    public void accessAPDU(APDU apdu) {
        try {
            apdu.setOutgoing();
        }catch (APDUException ae) {
        }
    }

    public void accessGlobalArray(byte[] array) {
        byte s = array[0];
        if (s == 0x00)
            array[1] = 0x01;
        else
            array[1] = 0x00;
    }

    public void accessAID(AID aid) {
        aid.equals(null);
    }

    public void parameterCheckAID(byte[] array, short mnum) {
        AID a = JCSystem.getAID();
        switch (mnum) {
        case 0:
        {
            boolean b = a.equals(array, ZERO, (byte)5);
        }
        return;
        case 1:
        {
            byte b = a.getBytes(array, ZERO);
        }
        return;
// case 2:
// { byte b = a.getPartialBytes(ZERO, array, ZERO, (byte)5); }
// return;
        case 3:
        {
            boolean b = a.partialEquals(array, ZERO, (byte)5);
        }
        return;
        case 4:
            a = new AID(array, ZERO, TESTAIDLEN);
            return;
        default:
            return;
        }
    }

    public boolean parameterCheckAID2(Object o) {
        return JCSystem.getAID().equals(o);
    }

    public boolean parameterCheckAID3(AID o) {
        return JCSystem.getAID().RIDEquals(o);
    }

    private static TestObject toc;
    private static TestObject tos;
    private static TestObject mtos;

    private static TestObjectShareable stoc;
    private static TestObjectShareable stos;
    private static TestObjectShareable smtos;

    public static void resetReferences() {
        toc = null;
        tos = null;
        stoc = null;
        stos = null;
        mtos = null;
        smtos = null;
    }

    public static void createTestObjectClient() {
        toc = new TestObject();
    }

    public static void createTestObjectServer() {
        tos = new TestObject();
    }

    public static void createTestObjectServerMultiselectable() {
        mtos = new TestObject();
    }

    public static void createShareableTestObjectClient() {
        stoc = new TestObjectShareable();
    }

    public static void createShareableTestObjectServer() {
        stos = new TestObjectShareable();
    }

    public static void createShareableTestObjectServerMultiselectable() {
        smtos = new TestObjectShareable();
    }

    public static TestObject getTestObjectClient() {
        return toc;
    }

    public static TestObject getTestObjectServer() {
        return tos;
    }

    public static TestObject getTestObjectServerMultiselectable() {
        return mtos;
    }

    public static TestObjectShareable getShareableTestObjectClient() {
        return stoc;
    }

    public static TestObjectShareable getShareableTestObjectServer() {
        return stos;
    }

    public static TestObjectShareable getShareableTestObjectServerMultiselectable() {
        return smtos;
    }

    public byte checkArray1(byte[] array) {
        byte result = array[0];
        return result;
    }

    public void checkArray2(byte[] array) {
        array[0] = 0x01;
    }

    public short checkArray3(byte[] array) {
        return (short)array.length;
    }

    public void throwException1(NullPointerException e){
        throw e;
    }

    public void throwException2(ISOException e) throws ISOException{
        throw e;
    }

    public void throwException3() {
        byte[] test = null;
        test[0] = 1;
    }

    public void callRecursive() {
        callRecursive();
    }

    public void accessCipher(byte[] buffer, boolean ext) {
        Cipher c = ext ? extCipher : intCipher;
        try {
            DESKey key = (DESKey)KeyBuilder.buildKey(KeyBuilder.TYPE_DES, KeyBuilder.LENGTH_DES, false);
            key.setKey(buffer, (short)0);
            c.init(key, Cipher.MODE_ENCRYPT);
            c.doFinal(buffer, (short)0, (short)8, buffer, (short)8);
        }catch (CryptoException ce) {
        }
    }

}
