// This file is part of the Java Card Firewall Tester program.
// 
// Authors: 
// 
// Wojciech Mostowski, woj@cs.ru.nl
// Erik Poll, erikpoll@cs.ru.nl
// Radboud University Nijmegen
// The Netherlands
// 
// Copyright (c) Wojciech Mostowski, Erik Poll,
// Radboud University Nijmegen (RU),
// Stichting Technische Wetenschappen (STW)
// 
// The Java Card Firewall Tester has been developed for the PinPas Java
// Card project, see http://www.win.tue.nl/pinpasjc/. The program is
// distributed under the licence terms that can be found in the LICENCE
// file in the main installation directory. Please refer to the LICENCE &
// README files for further information.

package sos.smartcards.firewallhost;

import java.util.*;
import javax.smartcardio.*;

public class TestSuite {

    Vector<OneTest> tests = new Vector<OneTest>();

    /** Test definitions, messages in human readable form. */
    TestSuite() {
	tests.add(new OneTest(
			      (byte)0x01,
			      false,
			      false,
			      new String[] {
				  "Global array install cannot be stored in an instance field.",
				  "Global array install cannot be stored in a static field.",
				  "APDU object cannot be stored in an instance field.",
				  "APDU object cannot be stored in a static field.",
				  "APDU buffer (global) cannot be stored in an instance field.",
				  "APDU buffer (global) cannot be stored in a static field.",
				  "System owned exception cannot be stored in an instance field.",
				  "System owned exception cannot be stored in a static field."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x02,
			      false,
			      false,
			      new String[] {
				  "Withing one context JCSystem.getAID returns applet's AID (1).",
				  "Withing one context JCSystem.getAID returns applet's AID (1).",
				  "Context does not change between applets in the same package.",
				  "Outside of the context JCSystem.getAID returns server's AID.",
				  "JCSystem.getPreviousContextAID returns context before the context switch.",
				  "JCSystem.getPreviousContextAID returns context before the context switch.",
				  "Context information has a stack structure.",
				  "AID is null before register is called (install method)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x03,
			      false,
			      false,
			      new String[] {
				  "CLEAR_ON_RESET array can be created within the same context.",
				  "CLEAR_ON_RESET array can be accessed within the same context.",
				  "CLEAR_ON_DESELECT array can be created within the same context.",
				  "CLEAR_ON_DESELECT array can be accessed within the same context."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x04,
			      true,
			      false,
			      new String[] {
				  "Persistent array can be created if the context is not the one currently selected.",
				  "CLEAR_ON_RESET array can be created if the context is not the one currently selected.",
				  "CLEAR_ON_DESELECT array cannot be created if the context is not the one currently selected.",
				  "Persistent array can be accessed if the context is not the one currently selected.",
				  "CLEAR_ON_RESET array can be accessed if the context is not the one currently selected.",
				  "CLEAR_ON_DESELECT array cannot be accessed if the context is not the one currently selected."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x05,
			      false,
			      false,
			      new String[] {
				  "Public static method calls do not cause context switch.",
				  "Public static fields are accessible accross contexts."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x06,
			      false,
			      false,
			      new String[] {
				  "Static access in applet package (methods and fields)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x07,
			      false,
			      false,
			      new String[] {
				  "Methods of APDU can be accessed from any context.",
				  "Global array (APDU buffer) can be accessed from any context.",
				  "Applet owned AID methods cannot be accessed from another context.",
				  "System owned AID methods can be accessed from any context.",
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x08,
			      false,
			      false,
			      new String[] {
				  "AID.equals checks for access rules to parameters.",
				  "AID.getBytes checks for access rules to parameters.",
				  "AID.getPartialBytes checks for access rules to parameters.",
				  "AID.partialEquals checks for access rules to parameters.",
				  "AID constructor checks for access rules to parameters.",
				  "AID.equals checks for access rules to parameters (arbitrary applet owned object).",
				  "AID.equals checks for access rules to parameters (applet owned AID).",
				  "AID.RIDEquals checks for access rules to parameters (applet owned AID).",
				  "AID.equals checks for access rules to parameters (system owned AID).",
				  "AID.RIDEquals checks for access rules to parameters (system owned AID)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x09,
			      false,
			      false,
			      new String[] {
				  "JCRE cannot read CLEAR_ON_DESELECT arrays if the current context is not the selected one (AID.equals).",
				  "JCRE cannot read CLEAR_ON_DESELECT arrays if the current context is not the selected one (AID.getBytes).",
				  "JCRE cannot read CLEAR_ON_DESELECT arrays if the current context is not the selected one (AID.getPartialBytes).",
				  "JCRE cannot read CLEAR_ON_DESELECT arrays if the current context is not the selected one (AID.partialEquals).",
				  "JCRE cannot read CLEAR_ON_DESELECT arrays if the current context is not the selected one (AID constructor).",
				  "JCRE can read CLEAR_ON_RESET arrays if the current context is not the selected one (AID.equals).",
				  "JCRE can read CLEAR_ON_RESET arrays if the current context is not the selected one (AID.getBytes).",
				  "JCRE can read CLEAR_ON_RESET arrays if the current context is not the selected one (AID.getPartialBytes).",
				  "JCRE can read CLEAR_ON_RESET arrays if the current context is not the selected one (AID.partialEquals).",
				  "JCRE can read CLEAR_ON_RESET arrays if the current context is not the selected one (AID constructor)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0A,
			      false,
			      false,
			      new String[] {
				  "JCRE can read CLEAR_ON_DESELECT arrays if the current context is the selected one (AID.equals).",
				  "JCRE can read CLEAR_ON_DESELECT arrays if the current context is the selected one (AID.getBytes).",
				  "JCRE can read CLEAR_ON_DESELECT arrays if the current context is the selected one (AID.getPartialBytes).",
				  "JCRE can read CLEAR_ON_DESELECT arrays if the current context is the selected one (AID.partialEquals).",
				  "JCRE can read CLEAR_ON_DESELECT arrays if the current context is the selected one (AID constructor)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0B,
			      false,
			      false,
			      new String[] {
				  "Static array belongs to the package (can be accessed within the same context)."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0C,
			      false,
			      false,
			      new String[] {
				  "Fields of object belonging to another context cannot be read.",
				  "Fields of object belonging to another context cannot be written.",
				  "Methods of object belonging to another context cannot be called.",
				  "Fields of shareable object belonging to another context cannot be read.",
				  "Fields of shareable object belonging to another context cannot be written.",
				  "Undeclared methods of shareable object belonging to another context cannot be called.",
				  "Declared methods of shareable object belonging to another context cannot be called without shareable cast.",
				  "Declared methods of shareable object belonging to another context can be called with shareable cast.",
				  "Fields of object belonging to current context can be read.",
				  "Fields of object belonging to current context can be written.",
				  "Methods of object belonging to current context can be called.",
				  "Fields of shareable object belonging to current context can be read.",
				  "Fields of shareable object belonging to current context can be written.",
				  "Undeclared methods of shareable object belonging to current context can be called."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0D,
			      false,
			      false,
			      new String[] {
				  "Context owned object can be cast to an interface type.",
				  "Context owned object can be cast to a class type.",
				  "Instance of context owned object can be checked against a class type.",
				  "Instance of context owned object can be checked against the Shareable interface.",
				  "Object owned by another context cannot be cast to a class type.",
				  "Instance of an object owned by another context cannot be checked against a class type.",
				  "Instance of an object owned by another context cannot be checked against the Shareable interface.",
				  "Context owned shareable object can be cast to an interface type.",
				  "Context owned shareable object can be cast to a class type.",
				  "Context owned shareable object can be cast to a shareable interface.",
				  "Instance of context owned shareable object can be checked against a class type.",
				  "Instance of context owned shareable object can be checked against the Shareable interface.",
				  "Instance of context owned shareable object can be checked against a shareable interface.",
				  "Shareable Object owned by another context cannot be cast to a non-shareable interface.",
				  "Shareable Object owned by another context cannot be cast to a class type.",
				  "Shareable Object owned by another context can be cast to a shareable interface.",
				  "Instance of a shareable object owned by another context cannot be checked against a class type.",
				  "Instance of a shareable object owned by another context can be checked against the Shareable interface.",
				  "Instance of a shareable object owned by another context can be checked against a shareable interface.",
				  "Instance of a system owned object (AID) can be checked against the Shareable interface.",
				  "Instance of a system owned object (AID) can be checked against a class type.",
				  "Instance of a system owned exception can be checked against a class type.",
				  "Instance of a system owned exception can be checked against the Shareable interface."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0E,
			      false,
			      false,
			      new String[] {
				  "An array cannot be read by another context.",
				  "An array cannot be written by another context.",
				  "Array field length cannot be read by another context."
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x0F,
			      false,
			      false,
			      new String[] {
				  "Context owned exceptions cannot be thrown by another context.",
				  "System owned exception can be thrown by another context.",
				  "Context is restored on exceptions occuring in another context (1).",      
				  "Context is restored on exceptions occuring in another context (2)."      
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x10,
			      false,
			      true,
			      new String[] {
				  "Shareable interface cannot be accessed if the server is selected on another channel."      
			      }
			      ));

	tests.add(new OneTest(
			      (byte)0x14,
			      true,
			      false,
			      new String[] {
				  "Shareable interface can be accessed if the multiselectable server is selected on another channel."      
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x11,
			      false,
			      true,
			      new String[] {
				  "Shareable interface cannot be acquired if the server is selected on another channel. Expected to fail (JC 2.2.2 only requirement)."      
			      }
			      ));

	tests.add(new OneTest(
			      (byte)0x15,
			      true,
			      false,
			      new String[] {
				  "Shareable interface can be acquired if the multiselectable server is selected on another channel."      
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x12,
			      false,
			      false,
			      new String[] {
				  "Stack overflow restores the context properly."      
			      }
			      ));
	tests.add(new OneTest(
			      (byte)0x13,
			      false,
			      false,
			      new String[] {
				  "Non-external ciphers should not be usable externally.",      
				  "External ciphers can be used externally."
			      }
			      ));
    }
  
    /** 
      * @param card the card object
      * @param channel the communication channel
      * @param sapplet the AID of the server applet
      * @param sappetms the AID of the multiselectable applet
      * @param jc21 indicates whether the test card is Java Card 2.1.* only
      */
    public void runTests(Card card, CardChannel channel, byte[] sapplet, byte[] sappletms, boolean jc21)  {
	Iterator<OneTest> it = tests.iterator();
    
	OneTest t = null;
	byte[] apdu = new byte[5];
	byte[] result = null;
	while(it.hasNext()) {
	    try {
		t = it.next();
		if(jc21 && (t.serverActive | t.serverMultiselectableActive)) continue;
		apdu[1] = t.testNum;
		if(jc21 | !t.serverActive) {
		    result = channel.transmit(new CommandAPDU(apdu)).getBytes();
		    analyzeResult(result, t.testNum, t.messages, "");
		}
		if(t.serverMultiselectableActive) {
		    CardChannel c = card.openLogicalChannel();
		    FirewallHost.selectAID(c, sappletms);
		    result = channel.transmit(new CommandAPDU(apdu)).getBytes();
		    c.close();
		    analyzeResult(result, t.testNum, t.messages, " Multiselectable server active.");
		}      

		if(t.serverActive) {
		    CardChannel c = card.openLogicalChannel();
		    FirewallHost.selectAID(c, sapplet);
		    result = channel.transmit(new CommandAPDU(apdu)).getBytes();
		    c.close();
		    analyzeResult(result, t.testNum, t.messages, "");
		}      
	    }catch(Exception e) {
		System.out.println("Test "+t.testNum+" failed: "+e);
	    }
	}
    }
  
    void analyzeResult(byte[] result, byte testNum, String[] m, String append) {
	byte[] r = new ResponseAPDU(result).getData();
	for(int i=0; i<r.length; i++) {
	    if(r[i] != 0x01) {
  	        String tNum = Integer.toHexString(testNum);
		if(tNum.length() == 1) tNum = "0" + tNum;
		System.out.println("Test failed: "+m[i]+append+
				   (r[i] != 0 ? (" Exception: "+decodeException(r[i])) : "") + 
				   (" INS: 0x"+tNum+" Test#: "+i)
				   );
	    }
	}
    }
  
    String decodeException(byte exCode) {
	switch(exCode) {
	case 0x21: return "ArithmeticException";
	case 0x22: return "ArrayStoreException";
	case 0x23: return "ClassCastException";
	case 0x24: return "ArrayIndexOutOfBoundsException";
	case 0x25: return "IndexOutOfBoundsException";
	case 0x26: return "NegativeArraySizeException";
	case 0x27: return "NullPointerException";
	case 0x28: return "SecurityException";
	case 0x31: return "RemoteException";
	case 0x41: return "UserException";
	case 0x51: return "APDUException";
	case 0x52: return "CryptoException";
	case 0x53: return "ISOException";
	case 0x54: return "PINException";
	case 0x55: return "ServiceException";
	case 0x56: return "SystemException";
	case 0x57: return "TransactionException";
	case (byte)0xAA: return "Unknown";
	default: return "Unknown";
	}
    }

}

class OneTest {

    byte testNum;
    boolean serverMultiselectableActive;
    boolean serverActive;
    String[] messages = null;
  
    /**
      * @param testNum test number INS byte
      * @param serverMultiselectableActive indicates whether the test should be also performed
      *        with the server (which is supposed to be multiselectable) active on another logical channel
      * @param serverActive indicates whether the server (non-multiselectable) should be active on another 
      *        logical channel
      * @param messages the array with messages in case any of the tests fails
      */
    OneTest(byte testNum, boolean serverMultiselectableActive, boolean serverActive, String[] messages) {
	this.testNum = testNum;
	this.serverMultiselectableActive = serverMultiselectableActive;
	this.serverActive = serverActive;
	this.messages = messages;
    }
  
}
