(* $Id: COrdFields.v,v 1.60 2000/11/10 11:27:27 freek Exp $ *)

Require Export CFields.
Require Export Refl_corr.
Require Export Step2.


(* ORDERED FIELDS *)

(* Tex_Prose
\section{Ordered Fields}
\subsection{Definition of the notion of ordered field}
*)

(* Begin_SpecReals *)

Implicit Arguments On.

(* Begin_Tex_Verb *)
Record strictorder [A:Set; R: A->A->Prop] : Prop :=
  { so_trans : (Transitive R);
    so_asym  : (antisymmetric R)
  }.
(* End_Tex_Verb *)

Implicit Arguments Off.

(* Begin_Tex_Verb *)
Record is_COrdField [F: CField; less : (CSetoid_relation F)] : Prop :=
 {ax_less_strorder : (strictorder less);
  ax_plus_resp_less: (x,y:F)(less x y) -> (z:F)(less (x[+]z) (y[+]z));
  ax_mult_resp_pos : (x,y:F)(less Zero x)->(less Zero y)->(less Zero (x[*]y));
  ax_less_conf_ap  : (x,y:F)(x [#] y) <-> ((less x y) \/ (less y x))
 }.
(* End_Tex_Verb *)

(* Begin_Tex_Verb *)
Record COrdField : Type :=
  {cof_crr   :> CField;
   cof_less  :  (CSetoid_relation cof_crr);
   cof_proof :  (is_COrdField cof_crr cof_less)
  }.
(* End_Tex_Verb *)

(* Tex_Prose
\begin{notation}
\verb!cof_less! is denoted infix by {\tt [:<]}.
The first argument is left implicit.
In the names of lemmas, {\tt [:<]} is written as \verb!less! and
 {\tt Zero [:<]} is written as \verb!pos!.
\end{notation}
*)
Syntactic Definition Less := (cof_less ?).
Infix NONA 8 "[:<]" Less.
Syntax constr level 8:
  less_infix [<<(csr_rel $_ (cof_less $_) $e1 $e2)>>] ->
    [[<hov 1> $e1:L [0 1] " [:<] " $e2:L]].

(* Begin_Tex_Verb *)
Definition greater := [F:COrdField; x,y : F] y[:<]x.
(* End_Tex_Verb *)

(* Tex_Prose
\begin{notation}
\verb!a [:>] b! denotes \verb!greater ? a b!.
\end{notation}
*)
Syntactic Definition Greater := (greater ?).
Infix NONA 8 "[:>]" Greater.
Syntax constr level 8:
  greater_infix [<<(greater $_ $e1 $e2)>>] ->
    [[<hov 1> $e1:L [0 1] " [:>] " $e2:L]].

(* End_SpecReals *)

(* Tex_Prose
Less or equal is defined as ``not greater than''.
*)

(* Begin_Tex_Verb *)
Definition leEq [F:COrdField; x,y:F]: Prop := ~(y[:<]x).
(* End_Tex_Verb *)

Syntactic Definition LeEq := (leEq ?).

(* Tex_Prose
\begin{notation}
{\tt LeEq} is denoted infix by {\tt [:<=]}.
In lemmas, {\tt [:<=]} is written as {\tt leEq}, and
{\tt Zero [:<=]} is written as \verb!nonneg!.
\end{notation}
*)
Infix NONA 8 "[:<=]" LeEq.
Syntax constr level 8:
  leEq_infix [<<(leEq $_ $e1 $e2)>>] ->
    [[<hov 1> $e1:L [0 1] " [:<=] " $e2:L]].

Section COrdField_axioms.
(* Tex_Prose
\subsection{COrdField axioms}
\begin{convention}
Let \verb!F! be a field.
\end{convention}
*)

Variable F : COrdField.

(* Begin_Tex_Verb *)
Lemma COrdField_is_COrdField : (is_COrdField F (cof_less F)).
(* End_Tex_Verb *)
Elim F; Intros.
Exact cof_proof0.
Qed.

(* Begin_Tex_Verb *)
Lemma less_strorder : (strictorder (cof_less F)).
(* End_Tex_Verb *)
Elim COrdField_is_COrdField; Intros.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma less_transitive_unfolded : (x,y,z:F)(x[:<]y) -> (y[:<]z) -> (x[:<]z).
(* End_Tex_Verb *)
Elim less_strorder.
Intros H1 H2.
Exact H1.
Qed.

(* Begin_Tex_Verb *)
Lemma less_antisymmetric_unfolded : (x,y:F)((x[:<]y) -> ~(y[:<]x)).
(* End_Tex_Verb *)
Elim less_strorder.
Intros H1 H2.
Exact H2.
Qed.

(* Begin_Tex_Verb *)
Lemma less_irreflexive : (irreflexive (cof_less F)).
(* End_Tex_Verb *)
Unfold irreflexive.
Intros.
Intro.
Apply (less_antisymmetric_unfolded ?? H H).
Qed.

(* Begin_Tex_Verb *)
Lemma less_irreflexive_unfolded : (x:F)(~(x[:<]x)).
(* End_Tex_Verb *)
Exact less_irreflexive.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_less_rht : (x,y,z:F)(x [:<] y) -> (x[+]z [:<] y[+]z).
(* End_Tex_Verb *)
Elim COrdField_is_COrdField; Intros.
Apply ax_plus_resp_less0.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_resp_pos :
                (x,y:F)(Zero [:<] x)->(Zero [:<] y)->(Zero [:<] (x[*]y)).
(* End_Tex_Verb *)
Elim COrdField_is_COrdField; Intros H1 H2 H3 H4.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma less_conf_ap : (x,y:F)(x [#] y) <-> ((x [:<] y) \/ (y [:<] x)).
(* End_Tex_Verb *)
Elim COrdField_is_COrdField; Intros H1 H2 H3 H4.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma less_wdr : (x,y,z:F)(x[:<]y) -> (y [=] z) -> (x[:<]z).
(* End_Tex_Verb *)
Exact (csr_wdr F (cof_less F)).
Qed.

(* Begin_Tex_Verb *)
Lemma less_wdl : (x,y,z:F)(x[:<]y) -> (x [=] z) -> (z[:<]y).
(* End_Tex_Verb *)
Exact (csr_wdl F (cof_less F)).
Qed.

(* Begin_Tex_Verb *)
Lemma less_wd_implies :
  (x,y,x',y':F)(x[=]x')->(y[=]y')->(implies x'[:<]y' x[:<]y).
(* End_Tex_Verb *)
Unfold implies.
Intros.
Apply less_wdl with x'.
Apply less_wdr with y'.
Assumption.
Apply eq_symmetric_unfolded.
Assumption.
Apply eq_symmetric_unfolded.
Assumption.
Qed.

Hints Resolve less_wd_implies: algebra_t.

End COrdField_axioms.

Tactic Definition Step_less_rht [$y] :=
  [<:tactic:<Apply less_wdr with $y;
    [Idtac | Algebra] >>].
Tactic Definition Step_less_lft [$y] :=
  [<:tactic:<Apply less_wdl with $y;
    [Idtac | Algebra] >>].

Tactic Definition Step_Rat_less_rht [$y] :=
  [<:tactic:<Apply less_wdr with $y;
    [Idtac | Rational] >>].
Tactic Definition Step_Rat_less_lft [$y] :=
  [<:tactic:<Apply less_wdl with $y;
    [Idtac | Rational] >>].

Section OrdField_basics.

(* Tex_Prose
\subsection{Basics}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.


(* Begin_Tex_Verb *)
Lemma less_imp_ap : (x,y:R)((x [:<] y) -> (x [#] y)).
(* End_Tex_Verb *)
Intros.
Elim (less_conf_ap ? x y); Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma Greater_imp_ap : (x,y:R)((y [:<] x) -> (x [#] y)).
(* End_Tex_Verb *)
Intros.
Elim (less_conf_ap ? x y); Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma ap_imp_less : (x,y:R)((x [#] y) -> ((x [:<] y)\/(y [:<] x))).
(* End_Tex_Verb *)
Intros.
Elim (less_conf_ap ? x y); Auto.
Qed.


(* Tex_Prose
   Now properties which can be derived.
*)

(* Begin_Tex_Verb *)
Lemma cotrans_less : (cotransitive (cof_less R)).
(* End_Tex_Verb *)
Unfold cotransitive.
Intros.
(Generalize (less_imp_ap ? ? H); Intro).
Elim (ap_cotransitive_unfolded ? ? ? H0 z).
Intro H1.
Elim (ap_imp_less ? ? H1).
Intro.
Left.
Assumption.
Intro H2.
Right.
Apply (less_transitive_unfolded ? ? ? ? H2 H).

Intro H1.
Elim (ap_imp_less ? ? H1).
Intro.
Right.
Assumption.

Intro H2.
Left.
Apply (less_transitive_unfolded ? ? ? ? H H2).
Qed.

(* Begin_Tex_Verb *)
Lemma cotrans_less_unfolded : (x,y:R)(x [:<] y)->(z:R)(x[:<]z)\/(z[:<]y).
(* End_Tex_Verb *)
Exact cotrans_less.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_ap_zero : (x:R)(Zero [:<] x)-> x [#] Zero.
(* End_Tex_Verb *)
Intros.
Apply Greater_imp_ap.
Assumption.
Defined.

End OrdField_basics.

(*
Implicit Arguments Off.
*)

Section infinity_of_cordfields.
(* Tex_Prose
\subsection{Infinity of ordered fields}

In an ordered field we have that \verb!One [+] One! and
\verb!One [+] One [+] One! and so on are all apart from zero.
We first show this, so that we can define \verb!TwoNZ!, \verb!ThreeNZ!
and so on. These are elements of \verb!NonZeros!, so that we can write
e.g.\ \verb!x[/]TwoNZ!.
*)

(* Tex_Prose
\begin{convention}
Let \verb!R! be a field.
\end{convention}
*)

Variable R : COrdField.

(* Begin_Tex_Verb *)
Lemma pos_one : (Zero::R [:<] One).
(* End_Tex_Verb *)
 (* 0[#]1, so 0<1 (and we are done) or 1<0; so assume 1<0. *)
Elim (ap_imp_less ??? (ring_non_triv R));[Idtac|Auto].
Intros.
 (* By plus_resp_less, 0=(1-1)<(0-1)=-1. *)
Cut (Zero::R) [:<] [--]One.
2:Step_less_lft One::R[+][--]One.
2:Step_less_rht Zero::R[+][--]One.
2:Apply plus_resp_less_rht.
Intros.
 (* By mult_resp_pos, 0<(-1).(-1)=1. *)
Step_Rat_less_rht [--]One::R[*][--]One.
Apply (mult_resp_pos ??? H0 H0).
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_less_succ : (m:nat)((Nring m)::R) [:<] (Nring (S m)).
(* End_Tex_Verb *)
Intro.
Simpl.
Step_less_rht (One [+] (nring R m)).
Step_less_lft (Zero [+] (nring R m)).
Apply plus_resp_less_rht.
Apply pos_one.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_less : (m,n:nat)(lt m n) -> (((Nring m)::R) [:<] (Nring n)).
(* End_Tex_Verb *)
Intros.
Elim H.
Apply nring_less_succ.
Intros.
Apply less_transitive_unfolded with (nring (R) m0).
Assumption.
Apply nring_less_succ.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_leEq : (m,n:nat)(le m n) -> (((Nring m)::R) [:<=] (Nring n)).
(* End_Tex_Verb *)
Intros.
Elim (le_lt_eq_dec ?? H); Intros.
Unfold leEq. Apply less_antisymmetric_unfolded.
Apply nring_less. Auto.
Rewrite y.
Unfold leEq. Apply less_irreflexive_unfolded.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_apart : (m,n:nat)(~(m=n) -> ((Nring m)::R [#] (Nring n))).
(* End_Tex_Verb *)
Intros.
Elim (nat_total_order ?? H); Intro.
Apply less_imp_ap.
Apply nring_less.
Assumption.
Apply Greater_imp_ap.
Apply nring_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_ap_zero :(n:nat)(~(n=O) ->((nring R n)[#]Zero)).
(* End_Tex_Verb *)
Intros.
Generalize (nring_apart ?? H); Intro.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma nringS_ap_zero:(m:nat)((nring R (S m))[#]Zero).
(* End_Tex_Verb *)
 Intros.
 Apply nring_ap_zero.
 Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_ap_zero_imp:(n:nat)((nring R n)[#]Zero)->(~(O=n)).
(* End_Tex_Verb *)
 Intros.
 Induction n.
 Intro.
 Unfold nring in H.
 Cut (~(Zero[#]Zero::R)).
 Intro.
 Apply H1.
 Assumption.
 Apply ap_irreflexive_unfolded.
 Apply O_S.
Qed.

(* Begin_Tex_Verb *)
Definition Snring : nat->(NonZeros R).
(* End_Tex_Verb *)
Intro n.
Apply (NZpro1 (nring R (S n))).
Unfold nonZeroP.
Apply nring_ap_zero.
Auto.
Defined.

Load Transparent_algebra.

(* Begin_Tex_Verb *)
Lemma pos_Snring : (n:nat)((Zero::R) [:<] (NZinj (Snring n))).
(* End_Tex_Verb *)
Intros.
Change (nring R O) [:<] (nring R (S n)).
Apply nring_less.
Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma nring_fac_ap_zero:(n:nat)((nring R (fac n))[#]Zero).
(* End_Tex_Verb *)
Proof.
Intro n;Apply nring_ap_zero. Cut (lt O (fac n)).
Omega.
Apply nat_fac_gtzero.
Qed.

Load Opaque_algebra.

Section up_to_four.

(* Tex_Prose
\subsubsection{Properties of one up to four}
\begin{nameconvention}
In the names of lemmas, we denote the numbers 0,1,2,3,4 and so on, by
\verb!zero!, \verb!one!, \verb!two! etc.
\end{nameconvention}
*)

(* Begin_Tex_Verb *)
Lemma less_plusOne: (x:R) x [:<] x[+]One.
(* End_Tex_Verb *)
 (* by plus_resp_less_rht and pos_one *)
Intros.
Step_less_lft Zero[+]x; Step_less_rht One[+]x.
Apply plus_resp_less_rht.
Exact pos_one.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_one_ext_less: (x,y:R)(x [:<] y) -> x [:<] y[+]One.
(* End_Tex_Verb *)
 (* By transitivity of less and less_plusOne *)
Intros.
Apply less_transitive_unfolded with y.
Assumption.
Apply less_plusOne.
Qed.

(* Begin_Tex_Verb *)
Lemma one_less_two : (One::R [:<] Two).
(* End_Tex_Verb *)
Simpl.
Step_less_rht One::R[+]One.
Apply less_plusOne.
Qed.

(* Begin_Tex_Verb *)
Lemma two_less_three : (Two::R [:<] Three).
(* End_Tex_Verb *)
Simpl.
Apply less_plusOne.
Qed.

(* Begin_Tex_Verb *)
Lemma three_less_four : (Three::R [:<] Four).
(* End_Tex_Verb *)
Simpl.
Apply less_plusOne.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_two : (Zero::R [:<] Two).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with One::R.
Exact pos_one.
Exact one_less_two.
Qed.

(* Begin_Tex_Verb *)
Lemma one_less_three : (One::R [:<] Three).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with (Two::R).
Exact one_less_two.
Exact two_less_three.
Qed.

(* Begin_Tex_Verb *)
Lemma two_less_four : (Two::R [:<] Four).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with (Three::R).
Exact two_less_three.
Exact three_less_four.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_three : (Zero::R [:<] Three).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with Two::R.
Exact pos_two.
Exact two_less_three.
Qed.


(* Begin_Tex_Verb *)
Lemma one_less_four : (One::R [:<] Four).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with (Three::R).
Exact one_less_three.
Exact three_less_four.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_four : (Zero::R [:<] Four).
(* End_Tex_Verb *)
Apply less_transitive_unfolded with Three::R.
Exact pos_three.
Exact three_less_four.
Qed.

(* Begin_Tex_Verb *)
Lemma two_ap_zero : Two [#] Zero::R.
(* End_Tex_Verb *)
Apply pos_ap_zero.
Apply pos_two.
Qed.

(* Begin_Tex_Verb *)
Lemma three_ap_zero : Three [#] Zero::R.
(* End_Tex_Verb *)
Apply pos_ap_zero.
Apply pos_three.
Qed.

(* Begin_Tex_Verb *)
Lemma four_ap_zero : Four [#] Zero::R.
(* End_Tex_Verb *)
Apply pos_ap_zero.
Apply pos_four.
Qed.

End up_to_four.

End infinity_of_cordfields.

(* Tex_Prose
\begin{convention}
\verb!OneNZ!, \verb!TwoNZ!, \verb!ThreeNZ! and \verb!FourNZ! are 1,
2, 3 and 4 as elements of \verb!NonZeros!.
\end{convention}
*)

Syntactic Definition OneNZ := One [//] (ring_non_triv ?).
Syntax constr level 1:
  oneNZ_constant [<<(nzpro $_ $_ (ring_non_triv $_))>>] -> ["OneNZ"].

Syntactic Definition TwoNZ := Two [//] (two_ap_zero ?).
Syntax constr level 1:
  twoNZ_constant [<<(nzpro $_ $_ (two_ap_zero $_))>>] -> ["TwoNZ"].

Syntactic Definition ThreeNZ := Three [//] (three_ap_zero ?).
Syntax constr level 1:
  threeNZ_constant [<<(nzpro $_ $_ (three_ap_zero $_))>>] -> ["ThreeNZ"].

Syntactic Definition FourNZ := Four [//] (four_ap_zero ?).
Syntax constr level 1:
  fourNZ_constant [<<(nzpro $_ $_ (four_ap_zero $_))>>] -> ["FourNZ"].

Section consequences_of_infinity.

(* Tex_Prose
\subsubsection{Consequences of infinity}
*)

(* Begin_Tex_Verb *)
Lemma square_eq :
   (F:COrdField)(x,a:F)(a [#] Zero)->
   (x[^](2) [=] a[^](2))->((x [=] a) \/ (x [=] [--]a)).
(* End_Tex_Verb *)
Intros.
Apply conditional_square_eq; Try Assumption.
Apply two_ap_zero.
Qed.


End consequences_of_infinity.

(**********************************************************************)
Section Basic_Properties_of_leEq.
(**********************************************************************)

(* Tex_Prose
\subsection{Properties of $\leq$ ({\tt leEq})}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.

(* Begin_Tex_Verb *)
Lemma leEq_wdr : (x,y,z:R)(x[:<=]y) -> (y [=] z) -> (x[:<=]z).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Apply H.
Apply less_wdl with z.
Assumption.
Algebra.
Qed.

(* Begin_Tex_Verb *)
Lemma leEq_wdl : (x,y,z:R)(x[:<=]y) -> (x [=] z) -> (z[:<=]y).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Apply H.
Apply less_wdr with z.
Assumption.
Algebra.
Qed.

(* Begin_Tex_Verb *)
Lemma leEq_reflexive : (x:R)(x[:<=]x).
(* End_Tex_Verb *)
Intro.
Unfold leEq.
Apply less_irreflexive_unfolded.
Qed.

(* Begin_Tex_Verb *)
Lemma eq_imp_leEq : (x,y:R)(x[=]y)->(x[:<=]y).
(* End_Tex_Verb *)
Intros x y H.
Apply leEq_wdr with x.
Exact (leEq_reflexive ?).
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma leEq_imp_eq : (x,y:R)(x [:<=] y) -> (y [:<=] x) -> (x [=] y).
(* End_Tex_Verb *)
Unfold leEq. Intros.
Apply not_ap_imp_eq. Intro. Apply H0.
Cut (x [:<] y) \/ (y [:<] x). Intro.
Elim H2; Clear H2; Intros. Auto.
Elim (H H2).
Apply ap_imp_less. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma lt_equiv_imp_eq : (x,x':R)
  ((y:R)(x [:<] y) -> (x' [:<] y)) ->
  ((y:R)(x' [:<] y) -> (x [:<] y)) ->
    x [=] x'.
(* End_Tex_Verb *)
Intros.
Apply leEq_imp_eq; Unfold leEq; Intro.
Cut x [:<] x. Fold ~(x [:<] x).
Apply less_antisymmetric_unfolded. Auto. Auto.
Cut x' [:<] x'. Fold ~(x' [:<] x').
Apply less_antisymmetric_unfolded. Auto. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma less_leEq_trans : (x,y,z:R)(x [:<] y) -> (y [:<=] z) -> (x [:<] z).
(* End_Tex_Verb *)
Intros x y z.
Unfold leEq.
Intros.
Elim (cotrans_less_unfolded ??? H z); Intro.
Assumption.
Elim H0.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma leEq_less_trans : (x,y,z:R)(x [:<=] y) -> (y [:<] z) -> (x [:<] z).
(* End_Tex_Verb *)
Intros x y z.
Unfold leEq.
Intros.
Elim (cotrans_less_unfolded ??? H0 x); Intro.
Elim H.
Assumption.
Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma leEq_transitive:(x,y,z:R)(x[:<=]y)->(y[:<=]z)->(x[:<=]z).
(* End_Tex_Verb *)
 Intros.
 Red.
 Intro.
 Apply H.
 Apply leEq_less_trans with y:=z.
 Assumption.
 Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma less_leEq : (x,y:R)(x [:<] y) -> (x [:<=] y).
(* End_Tex_Verb *)
Intros.
Unfold leEq.
Apply less_antisymmetric_unfolded.
Assumption.
Qed.

End Basic_Properties_of_leEq.

Tactic Definition Step_leEq_rht [$y] :=
  [<:tactic:<Apply leEq_wdr with $y;
    [Idtac | Algebra] >>].
Tactic Definition Step_leEq_lft [$y] :=
  [<:tactic:<Apply leEq_wdl with $y;
    [Idtac | Algebra] >>].

Tactic Definition Step_Rat_leEq_rht [$y] :=
  [<:tactic:<Apply leEq_wdr with $y;
    [Idtac | Rational] >>].
Tactic Definition Step_Rat_leEq_lft [$y] :=
  [<:tactic:<Apply leEq_wdl with $y;
    [Idtac | Rational] >>].


(**********************************************************************)
Section Properties_of_Ordering.
(**********************************************************************)

(* Tex_Prose
\subsection{Properties of $<$ ({\tt less})}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.


(* Tex_Prose
\begin{convention}
We do not use a special predicate for positivity,
(e.g. {\tt PosP}), but just write {\tt Zero [:<] x}.
Reasons: it is more natural; in ordinary mathematics we also write $0<x$
(or $x>0$).
\end{convention}
*)

Section addition.
(* Tex_Prose
\subsubsection{Addition and subtraction}\label{section:less_plus_minus}
\subsubsection*{$+$ and $-$ respect $<$}
*)

(* Begin_Tex_Verb *)
Lemma plus_resp_less_lft : (x,y,z:R)(x [:<] y) -> (z[+]x) [:<] (z[+]y).
(* End_Tex_Verb *)
Intros.
Step_less_lft x[+]z.
Step_less_rht y[+]z.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma min_resp_less : (x,y:R)(x [:<] y) -> ([--]y) [:<] ([--]x).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft x [+] ([--]x [+] [--]y).
Step_Rat_less_rht y [+] ([--]x [+] [--]y).
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma minus_resp_less : (x,y,z:R)(x [:<] y) -> ((x[-]z) [:<] (y[-]z)).
(* End_Tex_Verb *)
Transparent cg_minus.
Unfold cg_minus.
Intros.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma minus_resp_less_rht :(x,y,z:R)(y [:<] x) -> ((z[-]x) [:<] (z[-]y)).
(* End_Tex_Verb *)
Intros.
Transparent cg_minus.
Unfold cg_minus.
Apply plus_resp_less_lft.
Apply min_resp_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_less_both : (a,b,c,d:R)(a [:<] b) -> (c [:<] d) ->
                                       ((a[+]c) [:<] (b[+]d)).
(* End_Tex_Verb *)
Intros.
Apply less_transitive_unfolded with a[+]d.
Apply plus_resp_less_lft.
Assumption.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Tex_Prose
For versions of \verb!plus_resp_less_both! where one \verb![:<]! in the
assumption is replaced by \verb![:<=]! see
Section~\ref{section:leEq-plus-minus}.
*)

(* Tex_Prose
\subsubsection*{Cancellation laws}
*)

(* Begin_Tex_Verb *)
Lemma plus_cancel_less : (x,y,z:R)((x[+]z) [:<] (y[+]z)) -> (x [:<] y).
(* End_Tex_Verb *)
Intros.
(* Step_less_lft (x [+] Zero).
   Step_less_lft (x [+] (z [+] ([--] z))). *)
Step_Rat_less_lft ((x [+] z) [+] ([--] z)).
(* Step_less_rht (y [+] Zero).
   Step_less_rht (y [+] (z [+] ([--] z))). *)
Step_Rat_less_rht ((y [+] z) [+] ([--] z)).
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma min_cancel_less : (x,y:R)(([--]x) [:<] ([--]y)) -> (y [:<] x).
(* End_Tex_Verb *)
Intros.
Apply plus_cancel_less with ([--]x)[-]y.
Step_Rat_less_lft [--]x.
Step_Rat_less_rht [--]y.
Assumption.
Qed.

(* Tex_Prose
\subsubsection*{Laws for shifting}
Lemmas where an operation is transformed into the inverse operation on
the other side of an inequality are called laws for shifting.
\begin{nameconvention}
The names of laws for shifting start with ``\verb!shift_!'', and then come
the operation and the inequality, in the order in which they occur in the
conclusion.
If the shifted operand changes sides w.r.t. the operation and its inverse,
the name gets a prime.
\end{nameconvention}

It would be nicer to write the laws for shifting as bi-implications,
However, it is impractical to use these in Coq
(see the Coq shortcoming in Section~\ref{section:setoid-basics}).
*)

(* Begin_Tex_Verb *)
Lemma shift_less_plus :   (a,b,c:R)(a[-]c [:<] b) -> a [:<] b[+]c.
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft a[-]c[+]c.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_less_plus' :  (a,b,c:R)(a[-]b [:<] c) -> a [:<] b[+]c.
(* End_Tex_Verb *)
Intros.
Step_less_rht c[+]b.
Apply shift_less_plus.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_less_minus :  (a,b,c:R)(a[+]c [:<] b)-> (a [:<] b [-] c).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft (a[+]c)[-]c.
Apply minus_resp_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_less_minus' : (a,b,c:R)(c[+]a [:<] b)-> (a [:<] b [-] c).
(* End_Tex_Verb *)
Intros.
Apply shift_less_minus.
Step_less_lft c[+]a.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_plus_less :   (a,b,c:R)(a [:<] c [-] b) -> (a[+]b [:<] c).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_rht (c[-]b)[+]b.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_plus_less' :  (a,b,c:R)(b [:<] c [-] a) -> (a[+]b [:<] c).
(* End_Tex_Verb *)
Intros.
Step_less_lft b[+]a.
Apply shift_plus_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_minus_less :  (a,b,c:R)(a [:<] c[+]b) -> (a[-]b [:<] c).
(* End_Tex_Verb *)
Intros.
Step_less_rht (c[+]b)[-]b.
Apply minus_resp_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_minus_less' : (a,b,c:R)(a [:<] b[+]c) -> (a[-]b [:<] c).
(* End_Tex_Verb *)
Intros.
Apply shift_minus_less.
Step_less_rht b[+]c.
Assumption.
Qed.

(* Tex_Prose
Some special cases of laws for shifting.
*)

(* Begin_Tex_Verb *)
Lemma shift_zero_less_minus : (x,y:R)(x[:<]y) -> (Zero[:<](y[-]x)).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft x[-]x.
Apply minus_resp_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma qltone: (q:R) (q [:<] One) ->(q[-]One [#]Zero).
(* End_Tex_Verb *)
Intros.
Apply less_imp_ap.
Apply shift_minus_less.
Step_less_rht One::R.
Auto.
Qed.

End addition.

Section multiplication.
(* Tex_Prose
\subsubsection{Multiplication and division}
*)

(* Tex_Prose
   By convention~\ref{convention:div-form}
   in CFields (Section~\ref{section:fields}), we often have redundant premises
   in lemmas. E.g. the informal statement
     \[\text{for all }x, y \text{ in } R \text{ with } 0<x \text{ and } 0<y
       \text{ we have } 0 < y / x
     \]
   is formalized as follows.
\begin{verbatim}
        (x,y:R)(H: (x [#] Zero))(Zero [:<] x) -> (Zero [:<] y)->
        (Zero [:<] y[/]x[//]H)
\end{verbatim}
   We do this to keep it easy to use such lemmas.

\subsubsection*{Multiplication and division respect $<$}
*)

(* Begin_Tex_Verb *)
Lemma mult_resp_less :
 (x,y,z:R)(x [:<] y) -> (Zero [:<] z) -> ((x [*] z) [:<] (y [*] z)).
(* End_Tex_Verb *)
Intros.
Apply plus_cancel_less with ([--] (x[*]z)).
Step_less_lft (Zero::R).
(* Step_less_rht ((y[*]z) [-] (x[*]z)). *)
Step_Rat_less_rht ((y[-]x)[*]z).
Apply mult_resp_pos.
Step_less_lft (x[-]x).
Apply minus_resp_less.
Assumption.

Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma recip_resp_pos :
 (y:R)(Hy:y[#]Zero)(Zero [:<] y) -> Zero [:<] (One [/] y[//]Hy).
(* End_Tex_Verb *)
Intros.
Cut (Zero [:<] One[/]y[//]Hy) \/ (One[/]y[//]Hy [:<] Zero).
Intros. Elim H0; Clear H0; Intros.
Auto.
Cut One [:<] Zero::R. Intro.
Elim (less_antisymmetric_unfolded ??? (pos_one ?) H1).
Step_less_lft [--]([--]One::R). Step_less_rht [--]Zero::R.
Apply min_resp_less.
Step_Rat_less_rht y[*]([--](One[/]y[//]Hy)).
Apply mult_resp_pos. Auto.
Step_less_lft [--]Zero::R.
Apply min_resp_less. Auto.
Apply ap_imp_less.
Apply ap_symmetric_unfolded. Apply div_resp_ap_zero_rev.
Apply ring_non_triv.
Qed.


(* Begin_Tex_Verb *)
Lemma div_resp_less_rht :
 (x,y,z:R)(H1:z[#]Zero)(x [:<] y) -> (Zero [:<] z) ->
         ((x [/] z[//]H1) [:<] (y [/] z[//]H1)).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft x[*](One[/]z[//]H1).
Step_Rat_less_rht y[*](One[/]z[//]H1).
Apply mult_resp_less. Auto.
Apply recip_resp_pos.
Auto.
Qed.


(* Begin_Tex_Verb *)
Lemma div_resp_pos :
        (x,y:R)(H: (x [#] Zero))(Zero [:<] x) -> (Zero [:<] y)->
        (Zero [:<] y[/]x[//]H).
(* End_Tex_Verb *)
Intros.
Step_less_lft Zero[/]x[//]H.
Apply div_resp_less_rht; Auto.
Qed.


(* Begin_Tex_Verb *)
Lemma mult_resp_less_lft :
 (x,y,z:R)(x [:<] y) -> (Zero [:<] z) -> ((z [*] x) [:<] (z [*] y)).
(* End_Tex_Verb *)
Intros.
Step_less_lft (x [*] z).
Step_less_rht (y [*] z).
Apply mult_resp_less.
Assumption.
Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma mult_resp_less_both :
 (x,y,u,v:R)(Zero [:<=] x) -> (x [:<] y) -> (Zero [:<=] u) -> (u [:<] v) ->
 x[*]u [:<] y[*]v.
(* End_Tex_Verb *)
Cut (x,y,z:R)(x [:<=] y) -> (Zero [:<=] z) -> (x[*]z) [:<=] (y[*]z).
Intro resp_leEq.
Intros.
Apply leEq_less_trans with y[*]u.
Apply resp_leEq.
Apply less_leEq.
Assumption.
Assumption.
Apply mult_resp_less_lft.
Assumption.
Apply leEq_less_trans with x.
Assumption.
Assumption.

(* Cut *)
Unfold leEq.
Intros.
Intro.
Generalize (shift_zero_less_minus ?? H1); Intro.
Cut Zero [:<] (x[-]y)[*]z.
Intro.
2: Step_Rat_less_rht x[*]z[-]y[*]z.
2: Assumption.
Cut (a,b:R)(Zero [:<] a[*] b) ->
    ((Zero [:<] a) /\ (Zero [:<] b)) \/ ((a [:<] Zero) /\ (b [:<] Zero)).
Intros.
Generalize (H4 ?? H3); Intro.
Elim H5; Intro.
Elim H6; Intros.
Elim H.
Step_less_lft Zero[+]y.
Apply shift_plus_less.
Assumption.
Elim H6; Intros.
Elim H0.
Assumption.

Intros.
Generalize (Greater_imp_ap ??? H4); Intro.
Generalize (mult_cancel_ap_zero_lft ??? H5); Intro.
Generalize (mult_cancel_ap_zero_rht ??? H5); Intro.
Elim (ap_imp_less ??? H6); Intro.
Right.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
Assumption.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht ([--]a)[*]b.
Apply mult_resp_pos.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Left.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht a[*][--]b.
Apply mult_resp_pos.
Assumption.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma recip_resp_less :
 (x,y:R)(Hx:x[#]Zero)(Hy:y[#]Zero)(Zero [:<] x) -> (x [:<] y) ->
            (One [/] y[//]Hy) [:<] (One [/] x[//]Hx).
(* End_Tex_Verb *)
Intros.
Cut Zero [:<] x[*]y. Intro.
Cut x[*]y [#] Zero. Intro.
Step_Rat_less_lft x[*](One[/](x[*]y)[//]H2).
Step_Rat_less_rht y[*](One[/](x[*]y)[//]H2).
Apply mult_resp_less. Auto.
Apply recip_resp_pos. Auto.
Apply Greater_imp_ap. Auto.
Apply mult_resp_pos. Auto.
Apply less_transitive_unfolded with x; Auto.
Qed.


(* Tex_Prose
\subsubsection*{Cancellation laws}
*)
(* Begin_Tex_Verb *)
Lemma mult_cancel_less : (x,y,z:R)(Zero [:<] z) ->
                         ((x[*]z) [:<] (y[*]z)) -> (x [:<] y).
(* End_Tex_Verb *)
Intros.
Generalize (Greater_imp_ap ??? H); Intro.
Step_Rat_less_lft ((x[*]z)[*](One[/](z[//]H1))).
Step_Rat_less_rht ((y[*]z)[*](One[/](z[//]H1))).
Apply mult_resp_less.
Assumption.
Step_Rat_less_lft (Zero[/]z[//]H1).
Apply div_resp_less_rht.
Apply pos_one.
Assumption.
Qed.

(* Tex_Prose
\subsubsection*{Laws for shifting}
For namegiving, see the Section~\ref{section:less_plus_minus}
on plus and minus.
*)
(* Begin_Tex_Verb *)
Lemma shift_div_less : (x,y,z:R)(H:y[#]Zero)(Zero [:<] y) ->
                       (x [:<] (z[*]y)) -> ((x[/]y[//]H) [:<] z).
(* End_Tex_Verb *)
Intros.
Apply mult_cancel_less with y. Auto.
Step_less_lft x. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_div_less' : (x,y,z:R)(H:y[#]Zero)(Zero [:<] y) ->
                        (x [:<] (y [*]z)) -> ((x[/]y[//]H) [:<] z).
(* End_Tex_Verb *)
Intros.
Apply shift_div_less; Auto.
Step_less_rht y[*]z. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_less_div : (x,y,z:R)(H:y[#]Zero)(Zero [:<] y) ->
                    ((x[*]y) [:<] z) -> (x [:<] z[/]y[//]H).
(* End_Tex_Verb *)
Intros.
Apply mult_cancel_less with y. Auto.
Step_less_rht z. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_less_mult' : (x,y,z:R)(H:y[#]Zero)(Zero [:<] y) ->
                         ((x[/]y[//]H) [:<] z) -> (x [:<] (y [*]z)).
(* End_Tex_Verb *)
Intros.
Step_less_lft y[*](x[/]y[//]H).
Apply mult_resp_less_lft; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_mult_less : (x,y,z:R)(H:y[#]Zero)(Zero [:<] y) ->
                        (x [:<] z[/]y[//]H) -> ((x[*]y) [:<] z).
(* End_Tex_Verb *)
Intros.
Step_less_rht (z[/]y[//]H)[*]y.
Apply mult_resp_less; Auto.
Qed.

(* Tex_Prose
\subsubsection*{Other properties of multiplication and division}
*)
(* Begin_Tex_Verb *)
Lemma swap_div : (x,y,z:R)(Hy:y[#]Zero)(Hz:z[#]Zero)
                 (Zero [:<] y) -> (Zero [:<] z) ->
                 ((x[/]z[//]Hz)[:<] y) -> ((x[/]y[//]Hy) [:<] z).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_lft (x[/]z[//]Hz)[*](z[/]y[//]Hy).
Step_less_rht y[*](z[/]y[//]Hy).
Apply mult_resp_less. Auto.
Apply div_resp_pos; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma eps_div_less_eps : (eps,d:R)(d_:d [#] Zero)
                         (Zero[:<]eps) -> (One [:<] d) ->
                         ((eps[/](d[//]d_))[:<]eps).
(* End_Tex_Verb *)
Intros.
Apply shift_div_less'.
Apply less_transitive_unfolded with One::R.
Apply pos_one.

Assumption.

Step_less_lft (Cr_mult One eps).
Apply mult_resp_less.
Assumption.

Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_div_two : (eps:R)(Zero [:<] eps) -> (Zero [:<] (eps[/]TwoNZ)).
(* End_Tex_Verb *)
Intros.
Apply shift_less_div.
Apply pos_two.

Step_less_lft Zero::R.
Assumption.
Qed.


End multiplication.

Section misc.

(* Tex_Prose
\subsubsection{Miscellaneous}
*)

(* Begin_Tex_Verb *)
Lemma nring_pos : (m:nat)((lt O m) -> (Zero [:<] (nring R m))).
(* End_Tex_Verb *)
Intros. Induction m; Intros.
Elim (lt_n_n (0) H).
Elim m.
Simpl. Step_less_rht One::R. Apply pos_one.
Intros.
Step_less_rht (Nring (S n))[+]One::R.
Apply less_transitive_unfolded with Zero[+]One::R.
Step_less_rht One::R. Apply pos_one.
Apply plus_resp_less_rht. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma Smallest_less_Average : (a,b:R)(a[:<]b)->(a[:<]((a[+]b) [/] TwoNZ)).
(* End_Tex_Verb *)
Intros.
Apply shift_less_div.
Apply pos_two.
Step_Rat_less_lft a[+]a.
Apply plus_resp_less_lft.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma Average_less_Greatest : (a,b:R)(a[:<]b)->((a[+]b) [/] TwoNZ) [:<] b.
(* End_Tex_Verb *)
Intros.
Apply shift_div_less'.
Apply pos_two.
Step_Rat_less_rht b[+]b.
Apply plus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma sum_resp_less : (f,g:nat->R)(a,b:nat)(le a b)->
                      ((i:nat)(le a i) -> (le i b) -> (f i) [:<] (g i)) ->
                      (Sum a b f) [:<] (Sum a b g).
(* End_Tex_Verb *)
Intros.
Induction b; Intros.
Inversion H.
Unfold sum. Unfold sum1. Simpl.
Step_Rat_less_lft (f (0)). Step_Rat_less_rht (g (0)).
Apply H0. Auto. Auto.
Elim (le_lt_or_eq ?? H); Intros.
Apply less_wdl with (Sum a b f)[+](f (S b)).
Apply less_wdr with (Sum a b g)[+](g (S b)).
Apply plus_resp_less_both.
Apply Hrecb. Auto with arith. Auto.
Apply H0; Auto.
Apply eq_symmetric_unfolded. Apply sum_last.
Apply eq_symmetric_unfolded. Apply sum_last.
Rewrite H1.
Unfold sum. Unfold sum1. Simpl.
Step_Rat_less_lft (f (S b)).
Step_Rat_less_rht (g (S b)).
Apply H0; Auto.
Qed.

End misc.

End Properties_of_Ordering.


(**********************************************************************)
Section Properties_of_leEq.
(**********************************************************************)

(* Tex_Prose
\subsection{Properties of $\leq$ ({\tt leEq})}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)

Variable R : COrdField.

Section addition.
(* Tex_Prose
\subsubsection{Addition and subtraction}\label{section:leEq-plus-minus}
\subsubsection*{+ and - respect $\leq$}
*)

(* Begin_Tex_Verb *)
Lemma plus_resp_leEq : (x,y,z:R)(x [:<=] y) -> (x[+]z) [:<=] (y[+]z).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Apply H.
Apply (plus_cancel_less ? ? ? ? H0).
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_leEq_lft : (x,y,z:R)(x [:<=] y) -> (z[+]x) [:<=] (z[+]y).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft x[+]z.
Step_leEq_rht y[+]z.
Apply plus_resp_leEq; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma minus_resp_leEq : (x,y,z:R)(x [:<=] y) -> (x[-]z) [:<=] (y[-]z).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft x[+][--]z.
Step_leEq_rht y[+][--]z.
Apply plus_resp_leEq; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma min_resp_leEq :  (x,y:R)(x [:<=] y) -> ([--]y) [:<=] ([--]x).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Apply H.
Apply min_cancel_less.
Assumption.
Qed.



(* Begin_Tex_Verb *)
Lemma plus_resp_leEq_both : (x,y,a,b:R)(x [:<=] y)->(a [:<=] b)
	->(x [+] a [:<=] y [+] b).
(* End_Tex_Verb *)
 Intros.
 Apply leEq_transitive with y:=x [+] b.
 Step_Rat_leEq_lft a[+]x.
 Step_Rat_leEq_rht b[+]x.
 Apply plus_resp_leEq.
 Assumption.
 Apply plus_resp_leEq.
 Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_less_leEq : (a,b,c,d:R)(a [:<] b) -> (c [:<=] d) ->
                                       ((a[+]c) [:<] (b[+]d)).
(* End_Tex_Verb *)
Intros.
Apply leEq_less_trans with a[+]d.
Apply plus_resp_leEq_lft. Auto.
Apply plus_resp_less_rht. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_leEq_less : (a,b,c,d:R)(a [:<=] b) -> (c [:<] d) ->
                                       ((a[+]c) [:<] (b[+]d)).
(* End_Tex_Verb *)
Intros.
Step_less_lft c[+]a.
Step_less_rht d[+]b.
Apply plus_resp_less_leEq; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma minus_resp_less_leEq :
  (x,y,x',y':R)(x [:<=] y) -> (y' [:<] x') -> (x[-]x' [:<] y[-]y').
(* End_Tex_Verb *)
Intros.
Step_less_lft x[+][--]x'.
Step_less_rht y[+][--]y'.
Apply plus_resp_leEq_less.
Auto.
Apply min_resp_less. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma minus_resp_leEq_both :
  (x,y,x',y':R)(x [:<=] y) -> (y' [:<=] x') -> (x[-]x' [:<=] y[-]y').
(* End_Tex_Verb *)
Intros.
Step_leEq_lft x[+][--]x'.
Step_leEq_rht y[+][--]y'.
Apply plus_resp_leEq_both. Auto.
Apply min_resp_leEq. Auto.
Qed.

(* Tex_Prose
\subsubsection*{Cancellation properties}
*)
(* Begin_Tex_Verb *)
Lemma plus_cancel_leEq_rht: (x,y,z:R)( (x[+]z) [:<=] (y[+]z) )->
	(x[:<=]y).
(* End_Tex_Verb *)
 Intros.
 Step_Rat_leEq_lft x[+]z[+]([--]z).
 Step_Rat_leEq_rht y[+]z[+]([--]z).
 Apply plus_resp_leEq.
 Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma min_cancel_leEq:(x,y:R)([--]y[:<=][--]x)->(x[:<=]y).
(* End_Tex_Verb *)
 Intros.
 Step_Rat_leEq_lft [--][--]x.
 Step_Rat_leEq_rht [--][--]y.
 Apply min_resp_leEq.
 Assumption.
Qed.

(* Tex_Prose
\subsubsection*{Laws for shifting}
*)

(* Begin_Tex_Verb *)
Lemma shift_leEq_plus' : (a,b,c:R)(a[-]b [:<=] c) -> (a[:<=] b[+]c).
(* End_Tex_Verb *)
Intros.
Step_Rat_leEq_lft b[+](a[-]b).
Apply plus_resp_leEq_lft. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_leEq_rht: (a,b:R)(Zero [:<=] b[-]a) -> (a [:<=] b).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft Zero[+]a.
Step_Rat_leEq_rht (b[-]a)[+]a.
Apply plus_resp_leEq. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_leEq_lft : (a,b:R)(a [:<=] b) -> (Zero [:<=] b[-]a).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft a[-]a.
Apply minus_resp_leEq. Auto.
Qed.

End addition.

Section multiplication.
(* Tex_Prose
\subsubsection{Multiplication and division}
*)

(* Tex_Prose
\subsubsection*{Multiplication and division respect $\leq$}
*)

(* Begin_Tex_Verb *)
Lemma mult_resp_leEq_rht:
            (x,y,z:R)(x [:<=] y) -> (Zero [:<=] z) -> (x[*]z) [:<=] (y[*]z).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Generalize (shift_zero_less_minus ??? H1); Intro.
Cut Zero [:<] (x[-]y)[*]z.
Intro.
2: Step_Rat_less_rht x[*]z[-]y[*]z.
2: Assumption.
Cut (a,b:R)(Zero [:<] a[*] b) ->
    ((Zero [:<] a) /\ (Zero [:<] b)) \/ ((a [:<] Zero) /\ (b [:<] Zero)).
Intros.
Generalize (H4 ?? H3); Intro.
Elim H5; Intro.
Elim H6; Intros.
Elim H.
Step_less_lft Zero[+]y.
Apply shift_plus_less.
Assumption.
Elim H6; Intros.
Elim H0.
Assumption.

Intros.
Generalize (Greater_imp_ap ??? H4); Intro.
Generalize (mult_cancel_ap_zero_lft ??? H5); Intro.
Generalize (mult_cancel_ap_zero_rht ??? H5); Intro.
Elim (ap_imp_less ??? H6); Intro.
Right.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
Assumption.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht ([--]a)[*]b.
Apply mult_resp_pos.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Left.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht a[*][--]b.
Apply mult_resp_pos.
Assumption.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_resp_leEq_lft:
            (x,y,z:R)(x[:<=]y)->(Zero[:<=]z)-> (z[*]x[:<=]z[*]y).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft x[*]z.
Step_leEq_rht y[*]z.
Apply mult_resp_leEq_rht.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma recip_resp_leEq : (x,y:R)(Hx:x[#]Zero)(Hy:y[#]Zero)
                        (Zero[:<]y) -> (y[:<=]x) ->
                        (One[/]x[//]Hx [:<=] One[/]y[//]Hy).
(* End_Tex_Verb *)
Unfold leEq.
Intros. Intro. Apply H0.
Cut One[/]x[//]Hx [#] Zero. Intro Hx'.
Cut One[/]y[//]Hy [#] Zero. Intro Hy'.
Step_Rat_less_lft One[/](One[/]x[//]Hx)[//]Hx'.
Step_Rat_less_rht One[/](One[/]y[//]Hy)[//]Hy'.
Apply recip_resp_less.
Apply recip_resp_pos; Auto.
Auto.
Apply div_resp_ap_zero_rev. Apply ring_non_triv.
Apply div_resp_ap_zero_rev. Apply ring_non_triv.
Qed.

Hints Resolve recip_resp_leEq : algebra.

(* Tex_Prose
\subsubsection*{Cancellation properties}
*)

(* Begin_Tex_Verb *)
Lemma mult_cancel_leEq : (x,y,z:R)(Zero [:<] z) ->
                         ((x[*]z) [:<=] (y[*]z)) -> (x [:<=] y).
(* End_Tex_Verb *)
Unfold leEq.
Intros.
Intro.
Apply H0.
Apply mult_resp_less.
Assumption.
Assumption.
Qed.

(* Tex_Prose
\subsubsection*{Laws for shifting}
*)

(* Begin_Tex_Verb *)
Lemma shift_leEq_mult' :
                 (x,y,z:R)(H:y[#]Zero)(Zero[:<]y) ->
                 (x[/]y[//]H [:<=] z) -> (x [:<=] y[*]z).
(* End_Tex_Verb *)
Unfold leEq. Intros. Intro. Apply H1.
Apply shift_less_div. Auto.
Step_less_lft y[*]z. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma shift_leEq_div :
                 (x,y,z:R)(H:y[#]Zero)(Zero[:<]y) ->
                 (x[*]y [:<=] z) -> (x [:<=] z[/]y[//]H).
(* End_Tex_Verb *)
Unfold leEq. Intros. Intro. Apply H1.
Step_less_rht y[*]x.
Apply shift_less_mult' with H; Auto.
Qed.

Hints Resolve shift_leEq_div : algebra.

End multiplication.

Section misc.
(* Tex_Prose
\subsubsection{Miscellaneous}
*)

(* Begin_Tex_Verb *)
Lemma sqr_nonneg: (x:R) Zero [:<=] (x[^](2)).
(* End_Tex_Verb *)
Unfold leEq. Intros. Intro.
Cut Zero [:<] x[^](2). Intro.
Elim (less_antisymmetric_unfolded ??? H H0).
Cut (x [:<] Zero) \/ (Zero [:<] x). Intro. Elim H0; Clear H0; Intros.
Step_Rat_less_rht ([--]x)[*]([--]x).
Cut Zero [:<] [--]x. Intro.
Apply mult_resp_pos; Auto.
Step_less_lft [--]Zero::R. Apply min_resp_less. Auto.
Step_Rat_less_rht x[*]x.
Apply mult_resp_pos; Auto.
Apply ap_imp_less.
Apply cring_mult_ap_zero with x.
Step_ap_lft x[^](2).
Apply less_imp_ap. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma cc_abs_aid: (x,y:R)(Zero [:<=] ((x[^](2)) [+] (y[^](2)))).
(* End_Tex_Verb *)
Intros.
Step_leEq_lft Zero[+]Zero::R.
Apply plus_resp_leEq_both; Apply sqr_nonneg.
Qed.

Load Transparent_algebra.

(* Begin_Tex_Verb *)
Lemma nexp_resp_pos : (x:R)(k:nat)(Zero [:<] x)->(Zero [:<] x[^]k).
(* End_Tex_Verb *)
Intros.
Elim k.
Simpl.
Apply pos_one.
Intros.
Simpl.
Apply mult_resp_pos.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma power_resp_leEq : (x,y:R)(k:nat)
  (Zero [:<] x)->(x [:<=] y)->(x[^]k [:<=] y[^]k).
(* End_Tex_Verb *)
Intros. Induction k; Intros.
Step_leEq_lft One::R.
Step_leEq_rht One::R.
Apply leEq_reflexive.
Step_leEq_lft x[^]k[*]x.
Step_leEq_rht y[^]k[*]y.
Apply leEq_transitive with x[^]k[*]y.
Apply mult_resp_leEq_lft. Auto.
Apply less_leEq.
Apply nexp_resp_pos; Auto.
Apply mult_resp_leEq_rht. Auto.
Apply less_leEq. Apply less_leEq_trans with x; Auto.
Qed.

Load Opaque_algebra.

(* Begin_Tex_Verb *)
Lemma mult_resp_nonneg : (x,y:R)(Zero[:<=]x) -> (Zero[:<=]y) ->
                                (Zero[:<=] x[*]y).
(* End_Tex_Verb *)
Unfold leEq. Intros. Intro. Apply H0.
Cut x[*]y [#] Zero. Intro.
Cut x [#] Zero. Intro.
Cut y [#] Zero. Intro.
Elim (ap_imp_less ??? H4); Intros. Auto.
Elim (ap_imp_less ??? H3); Intros. Elim (H H6).
Elim (less_antisymmetric_unfolded ??? H1 (mult_resp_pos ??? H6 H5)).
Apply cring_mult_ap_zero_op with x. Auto.
Apply cring_mult_ap_zero with y. Auto.
Apply less_imp_ap. Auto.
Qed.

Load Transparent_algebra.

(* Begin_Tex_Verb *)
Lemma nexp_resp_nonneg : (x:R)(k:nat)(Zero [:<=] x)->(Zero [:<=] x[^]k).
(* End_Tex_Verb *)
Intros. Induction k. Intros.
Step_leEq_rht One::R. Apply less_leEq. Apply pos_one.
Step_leEq_rht x[^]k[*]x.
Apply mult_resp_nonneg; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma nexp_resp_less : (x,y:R)(n:nat)(le (1) n) -> (Zero [:<=] x) ->
                             (x [:<] y) -> x[^]n [:<] y[^]n.
(* End_Tex_Verb *)
Intros.
Induction n.
ElimType False.
Omega.
Elim n.
Simpl.
Step_less_lft x.
Step_less_rht y.
Assumption.
Intros.
Change x[^](S n0) [*] x [:<] y[^](S n0) [*] y.
Apply mult_resp_less_both.
Apply nexp_resp_nonneg.
Assumption.
Assumption.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma power_cancel_leEq : (x,y:R)(k:nat)(lt O k) ->
  (Zero [:<=] y)->(x[^]k [:<=] y[^]k)->(x [:<=] y).
(* End_Tex_Verb *)
Unfold leEq. Intros. Intro. Apply H1.
Apply nexp_resp_less; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma power_cancel_less : (x,y:R)(k:nat)
  (Zero [:<=] y)->(x[^]k [:<] y[^]k)->(x [:<] y).
(* End_Tex_Verb *)
Intros.
Elim (zerop k); Intros.
Rewrite y0 in H0.
Cut One [:<] One::R. Intro.
Elim (less_irreflexive_unfolded ?? H1).
Step_less_lft x[^](0). Step_less_rht y[^](0). Auto.
Cut (x [:<] y) \/ (y [:<] x). Intro.
Elim H1; Clear H1; Intros. Auto.
Cut (x [:<=] y). Intro. Elim (H2 H1).
Apply power_cancel_leEq with k; Auto.
Apply less_leEq. Auto.
Apply ap_imp_less. Apply un_op_strext_unfolded with (nexp_op R k).
Apply less_imp_ap. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma sum_resp_leEq : (f,g:nat->R)(a,b:nat)(le a (S b)) ->
                      ((i:nat)(le a i) -> (le i b) -> (f i) [:<=] (g i)) ->
                      (Sum a b f) [:<=] (Sum a b g).
(* End_Tex_Verb *)
Intros. Induction b; Intros.
Unfold sum. Unfold sum1.
Inversion H.
Step_leEq_lft Zero::R.
Step_leEq_rht Zero::R.
Apply leEq_reflexive.
Inversion H2.
Simpl.
Step_Rat_leEq_lft (f (0)).
Step_Rat_leEq_rht (g (0)).
Apply H0; Auto.
Elim (le_lt_or_eq ?? H); Intros.
Apply leEq_wdl with (Sum a b f)[+](f (S b)).
Apply leEq_wdr with (Sum a b g)[+](g (S b)).
Apply plus_resp_leEq_both.
Apply Hrecb. Auto with arith. Auto.
Apply H0. Auto with arith. Auto.
Apply eq_symmetric_unfolded. Apply sum_last.
Apply eq_symmetric_unfolded. Apply sum_last.
Unfold sum. Unfold sum1.
Rewrite H1.
Simpl.
Step_leEq_lft Zero::R.
Step_leEq_rht Zero::R.
Apply leEq_reflexive.
Qed.

(* Begin_Tex_Verb *)
Lemma approach_zero:(x:R)((e:R)(Zero[:<]e)->(x[:<]e))->
			~(Zero[:<]x).
(* End_Tex_Verb *)
 Intros.
 Intro.
 Cut (x[:<](x[/]TwoNZ)).
 Change (~(x [:<] x[/]TwoNZ)).
 Apply less_antisymmetric_unfolded.
 Apply plus_cancel_less with z:=[--]x[/]TwoNZ.
 Apply mult_cancel_less with z:=Two::R.
 Apply pos_two.
 Step_Rat_less_lft Zero::R.
 Step_Rat_less_rht x.
 Assumption.
 Apply H.
 Apply pos_div_two.
 Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma approach_zero_weak:(x:R)((e:R)(Zero [:<] e)-> x [:<=] e)->
                              ~(Zero [:<] x).
(* End_Tex_Verb *)
 Intros.
 Intro.
 Cut (x[:<=](x[/]TwoNZ)).
 Change (~~(x[/]TwoNZ [:<] x)).
 Intro.
 Apply H1.
 Apply plus_cancel_less with z:=[--]x[/]TwoNZ.
 Apply mult_cancel_less with z:=Two::R.
 Apply pos_two.
 Step_Rat_less_lft Zero::R.
 Step_Rat_less_rht x.
 Assumption.
 Apply H.
 Apply pos_div_two.
 Assumption.
Qed.
End misc.

End Properties_of_leEq.

(**********************************************************************)
Section PosP_properties.
(**********************************************************************)

(* Tex_Prose
\subsection{Properties of positive numbers}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.

(* Begin_Tex_Verb *)
Syntactic Definition ZeroR := (Zero::R).
Syntactic Definition OneR := (One::R).
(* End_Tex_Verb *)

(* Begin_Tex_Verb *)
Definition pos_ap_zeros: (x:R)(Zero [:<] x)-> (NonZeros R).
(* End_Tex_Verb *)
Intros.
Apply (nzpro ? x).
Apply pos_ap_zero.
Assumption.
Defined.

(* Begin_Tex_Verb *)
Lemma mult_pos_imp : (a,b:R)(Zero [:<] a[*] b) ->
    ((Zero [:<] a) /\ (Zero [:<] b)) \/ ((a [:<] Zero) /\ (b [:<] Zero)).
(* End_Tex_Verb *)
Generalize I; Intro.
Generalize I; Intro.
Generalize I; Intro.
Generalize I; Intro.
Generalize I; Intro.
Intros.
Generalize (Greater_imp_ap ??? H4); Intro.
Generalize (mult_cancel_ap_zero_lft ??? H5); Intro.
Generalize (mult_cancel_ap_zero_rht ??? H5); Intro.
Elim (ap_imp_less ??? H6); Intro.
Right.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
Assumption.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht ([--]a)[*]b.
Apply mult_resp_pos.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Left.
Split.
Assumption.
Elim (ap_imp_less ??? H7); Intro.
ElimType False.
Apply (less_irreflexive_unfolded R Zero).
Apply less_transitive_unfolded with a[*]b.
Assumption.
Apply min_cancel_less.
Step_less_lft Zero::R.
Step_less_rht a[*][--]b.
Apply mult_resp_pos.
Assumption.
Step_less_lft [--]Zero::R.
Apply min_resp_less.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_pos_nonneg :
       (x,y:R)(Zero [:<] x) -> (Zero [:<=] y) -> (Zero [:<] x[+]y).
(* End_Tex_Verb *)
Intros.
Apply less_leEq_trans with x. Auto.
Step_leEq_lft x[+]Zero.
Apply plus_resp_leEq_lft. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_nonneg_pos :
       (x,y:R)(Zero [:<=] x) -> (Zero [:<] y) -> (Zero [:<] x[+]y).
(* End_Tex_Verb *)
Intros.
Step_less_rht y[+]x.
Apply plus_resp_pos_nonneg; Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_square : (x:R)(x [#] Zero)->(Zero [:<] x[^](2)).
(* End_Tex_Verb *)
Intros.
Cut (x [:<] Zero) \/ (Zero [:<] x). Intro. Elim H0; Clear H0; Intros.
Step_Rat_less_rht ([--]x)[*]([--]x).
Cut Zero [:<] [--]x. Intro.
Apply mult_resp_pos; Auto.
Step_less_lft [--]Zero::R.
Apply min_resp_less. Auto.
Step_Rat_less_rht x[*]x.
Apply mult_resp_pos; Auto.
Apply ap_imp_less. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_cancel_pos_rht : (x,y:R)(Zero [:<] x[*]y)->(Zero [:<] x)->(Zero [:<] y).
(* End_Tex_Verb *)
Intros.
Cut x [#] Zero. Intro.
Step_Rat_less_rht (x[*]y)[*](One[/]x[//]H1).
Apply mult_resp_pos. Auto.
Apply recip_resp_pos. Auto.
Apply Greater_imp_ap. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_cancel_pos_lft : (x,y:R)(Zero [:<] x[*]y)->(Zero [:<] y)->(Zero [:<] x).
(* End_Tex_Verb *)
Intros.
Apply mult_cancel_pos_rht with y.
Step_less_rht x[*]y.
Auto. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_wd : (x,y:R)(x [=] y)->(Zero [:<] y)->(Zero [:<] x).
(* End_Tex_Verb *)
Intros.
Step_less_rht y.
Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma even_power_pos : (n:nat)(even n) -> (x:R)(x [#] Zero) -> (Zero [:<] x[^]n).
(* End_Tex_Verb *)
Intros.
Elim (even_2n n H). Intro m. Intros.
Replace n with (mult (2) m).
Step_less_rht (x[^](2))[^]m.
Apply nexp_resp_pos.
Apply pos_square. Auto.
Rewrite y. Unfold double. Omega.
Qed.

(* Begin_Tex_Verb *)
Lemma odd_power_cancel_pos : (n:nat)(odd n)->(x:R)(Zero [:<] x[^]n)->(Zero [:<] x).
(* End_Tex_Verb *)
Intros.
Induction n.
Inversion H.
Apply mult_cancel_pos_rht with x[^]n.
Step_less_rht x[^](S n). Auto.
Apply even_power_pos.
Inversion H. Auto.
Apply un_op_strext_unfolded with (nexp_op R (S n)).
Cut (lt (0) (S n)). Intro.
Step_ap_rht Zero::R.
Apply Greater_imp_ap. Auto.
Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma plus_resp_pos : (x,y:R)(Zero [:<] x)->(Zero [:<] y)->(Zero [:<] x[+]y).
(* End_Tex_Verb *)
Intros.
Apply plus_resp_pos_nonneg.
Auto.
Apply less_leEq. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma pos_nring_S: (n:nat)((Zero::R) [:<] (Nring (S n))).
(* End_Tex_Verb *)
Induction n; Simpl; Intros.
(* base *)
Step_less_rht (One::R); Apply pos_one.
(* step *)
Apply less_transitive_unfolded with (nring R n0)[+]One.
Assumption.
Apply less_plusOne.
Qed.

End PosP_properties.

(**********************************************************************)
Section nonneg_properties.
(**********************************************************************)

(* Tex_Prose
\subsection{Properties of non-negative numbers}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.

(* Begin_Tex_Verb *)
Syntactic Definition ZeroR := (Zero::R).
Syntactic Definition OneR := (One::R).
(* End_Tex_Verb *)


End nonneg_properties.

Hints Resolve mult_resp_nonneg.

(* Tex_Prose
\subsection{Properties of {\tt AbsSmall}.}
*)

(* Begin_SpecReals *)

(* Begin_Tex_Verb *)
Definition absSmall [R:COrdField;e,x:R]: Prop := ([--]e [:<] x) /\ (x [:<] e).
(* End_Tex_Verb *)

Syntactic Definition AbsSmall := (absSmall ?).

(* End_SpecReals *)

(**********************************************************************)
Section AbsSmall_wd_properties.
(**********************************************************************)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.

(* Begin_Tex_Verb *)
Lemma AbsSmall_wd_rht : (rel_well_def_rht R AbsSmall).
(* End_Tex_Verb *)
Unfold rel_well_def_rht.
Unfold AbsSmall.
Intros.
(Elim H; Intros).
Split.
Step_less_rht y.
Assumption.

Step_less_lft y.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_wd_rht_unfolded : (x,y,z:R)(AbsSmall x y)->(y[=]z)->
                                 (AbsSmall x z).
(* End_Tex_Verb *)
Proof AbsSmall_wd_rht.

(* Begin_Tex_Verb *)
Lemma AbsSmall_wd_lft : (rel_well_def_lft R AbsSmall).
(* End_Tex_Verb *)
Unfold rel_well_def_lft.
Unfold absSmall.
Intros.
(Elim H; Intros).
Split.
Step_less_lft ([--]x).
Assumption.

Step_less_rht x.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_wd_lft_unfolded : (x,y,z:R)(AbsSmall x y)->(x[=]z)->
                                 (AbsSmall z y).
(* End_Tex_Verb *)
Proof AbsSmall_wd_lft.

End AbsSmall_wd_properties.

Tactic Definition Step_AbsSmall_rht [$y] :=
  [<:tactic:<Apply AbsSmall_wd_rht_unfolded with $y;
    [Idtac | Algebra] >>].
Tactic Definition Step_AbsSmall_lft [$y] :=
  [<:tactic:<Apply AbsSmall_wd_lft_unfolded with $y;
    [Idtac | Algebra] >>].

Tactic Definition Step_Rat_AbsSmall_rht [$y] :=
  [<:tactic:<Apply AbsSmall_wd_rht_unfolded with $y;
    [Idtac | Rational] >>].
Tactic Definition Step_Rat_AbsSmall_lft [$y] :=
  [<:tactic:<Apply AbsSmall_wd_lft_unfolded with $y;
    [Idtac | Rational] >>].

(**********************************************************************)
Section AbsSmall_properties.
(**********************************************************************)

(* No new Latex section; we only start a new Coq section because the
   tactic definitions above shouldn't be inside a section. *)

Variable R : COrdField.

(* Begin_Tex_Verb *)
Syntactic Definition ZeroR := (Zero::R).
(* End_Tex_Verb *)



(* Begin_Tex_Verb *)
Lemma AbsSmall_leEq_trans :  (e1,e2,d:R)(e1 [:<=] e2) ->
                             (AbsSmall e1 d) -> (AbsSmall e2 d).
(* End_Tex_Verb *)
Unfold absSmall.
Intros.
(Elim H0; Intros).
Split.
Apply leEq_less_trans with ([--]e1).
Apply min_resp_leEq.
Assumption.

Assumption.

Apply less_leEq_trans with e1.
Assumption.

Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_trans :  (e1,e2,d:R)(e1 [:<] e2) ->
                        (AbsSmall e1 d) -> (AbsSmall e2 d).
(* End_Tex_Verb *)
Intros.
Apply AbsSmall_leEq_trans with e1.
Apply less_leEq.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma less_imp_AbsSmall : (e,d:R)(Zero [:<=] e) -> (e [:<] d) ->
                           (AbsSmall d e).
(* End_Tex_Verb *)
Intros.
Unfold absSmall.
Split; Try Assumption.
Apply less_leEq_trans with ZeroR; Try Assumption.
Step_less_rht [--]ZeroR.
Apply min_resp_less.
Apply leEq_less_trans with e; Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma min_resp_AbsSmall: (x,y:R)(AbsSmall x y) -> (AbsSmall x [--]y).
(* End_Tex_Verb *)
Unfold absSmall.
Intros.
Elim H; Intros.
Split.
Apply min_resp_less.
Assumption.
Step_less_rht [--]([--]x).
Apply min_resp_less.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_minus : (e,x1,x2:R)
                       (AbsSmall e (x1[-]x2)) -> (AbsSmall e (x2[-]x1)).
(* End_Tex_Verb *)
Intros.
Step_Rat_AbsSmall_rht [--](x1[-]x2).
Apply min_resp_AbsSmall.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_plus : (e1,e2,x1,x2:R)
                      (AbsSmall e1 x1) -> (AbsSmall e2 x2) ->
                      (AbsSmall (e1[+]e2) (x1[+]x2)).
(* End_Tex_Verb *)
Unfold absSmall.
Intros.
Elim H; Intros.
Elim H0; Intros.
Split.
Step_Rat_less_lft ([--]e1) [+] ([--]e2).
Apply plus_resp_less_both; Assumption.
Apply plus_resp_less_both; Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_eps_div_two : (e,x1,x2:R)
                             (AbsSmall (e[/]TwoNZ) x1) ->
                             (AbsSmall (e[/]TwoNZ) x2) ->
                             (AbsSmall e (x1[+]x2)).
(* End_Tex_Verb *)
Intros.
Step_Rat_AbsSmall_lft (e[/]TwoNZ [+] e[/]TwoNZ).
Apply AbsSmall_plus.
Assumption.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_x_plus_delta : (x,eps,delta:R)
               (Zero [:<] eps) -> (Zero [:<] delta) -> (delta[:<]eps) ->
               (AbsSmall eps (x[-](x[+]delta))).
(* End_Tex_Verb *)
Intros.
(* Step_AbsSmall_rht ((x[-]x)[-]delta).
Step_AbsSmall_rht (Zero [-] delta). *)
Step_Rat_AbsSmall_rht ([--] delta).
Apply min_resp_AbsSmall.
Apply less_imp_AbsSmall.
Apply less_leEq; Assumption.
Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma AbsSmall_x_minus_delta : (x,eps,delta:R)
                  (Zero [:<] eps) -> (Zero[:<] delta) -> (delta[:<]eps) ->
                  (AbsSmall eps (x[-](x[-]delta))).
(* End_Tex_Verb *)
Intros.
(* Step_AbsSmall_rht ((x[-]x)[+]delta).
   Step_AbsSmall_rht (Zero [+] delta). *)
Step_Rat_AbsSmall_rht delta.
Apply less_imp_AbsSmall.
Apply less_leEq; Assumption.
Assumption.
Qed.






(* Begin_Tex_Verb *)
Lemma AbsSmall_x_plus_eps_div2 : (x,eps:R)(Zero [:<] eps)->
                    (AbsSmall eps (x[-](x[+]eps[/]TwoNZ))).
(* End_Tex_Verb *)
Intros.
Apply AbsSmall_x_plus_delta.
Assumption.

Apply pos_div_two.
Assumption.

Unfold pos_ap_zeros.
Apply eps_div_less_eps.
Assumption.

Apply one_less_two.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_x_minus_eps_div2 : (x,eps:R)(Zero [:<] eps)->
                     (AbsSmall eps (x[-](x[-]eps[/]TwoNZ))).
(* End_Tex_Verb *)
Intros.
Apply AbsSmall_x_minus_delta.
Assumption.

Apply pos_div_two.
Assumption.

Unfold pos_ap_zeros.
Apply eps_div_less_eps.
Assumption.

Apply one_less_two.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_intermediate : (x,y,z,eps:R)(x [:<] y)->(y[:<]z)->
                              (AbsSmall eps (z[-]x)) -> (AbsSmall eps (z[-]y)).
(* End_Tex_Verb *)
Intros.
Apply less_imp_AbsSmall.
Apply less_leEq.
Apply shift_zero_less_minus.
Assumption.
Unfold absSmall in H1.
Elim H1; Intros.
Apply less_transitive_unfolded with z[-]x; Try Assumption.
Apply minus_resp_less_rht.
Assumption.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_eps_div2 : (eps:R)(Zero [:<] eps)->
                          (AbsSmall eps (eps[/]TwoNZ)).
(* End_Tex_Verb *)
Intros.
Apply less_imp_AbsSmall.
Apply less_leEq.
Apply pos_div_two.
Assumption.

Apply eps_div_less_eps.
Assumption.

Apply one_less_two.
Qed.

(* Begin_Tex_Verb *)
Lemma AbsSmall_pos: (e,x:R)(AbsSmall e x)->(Zero[:<]e).
(* End_Tex_Verb *)
 Unfold absSmall.
 Intros.
 Elim H.
 Intros.
 Cut ([--]e [:<] e).
 Intros.
 Apply mult_cancel_less with z:=Two::R.
 Apply pos_two.
 Apply plus_cancel_less with z:=[--]e.
 Step_Rat_less_lft [--]e.
 Step_Rat_less_rht e.
 Assumption.
 Apply less_transitive_unfolded with y:=x.
 Assumption.
 Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma AbsSmall_mult :(e1,e2,x1,x2: R)(AbsSmall e1 x1)->
	(AbsSmall e2 x2)->(AbsSmall Three[*](e1[*]e2) x1[*]x2).
(* End_Tex_Verb *)
 Unfold absSmall.
 Intros.
 Elim H.
 Intros.
 Elim H0.
 Intros.
 Cut (Zero[:<]e1).
 Intro.
 Cut (Zero[:<]e2).
 Intro.
 Split.

 Apply plus_cancel_less with z:=(Three[*](e1[*]e2)).
 Step_Rat_less_lft Zero::R.
 Step_Rat_less_rht x1[*]x2[+](e1[*]e2)[+](e1[*]e2)[+](e1[*]e2).
 Apply less_transitive_unfolded with
   	 y:= x1[*]x2[+](e1[*]e2)[+](x1[*]e2)[+](e1[*]x2).
 Step_Rat_less_rht (e1[+]x1)[*](e2[+]x2).
 Apply mult_resp_pos.
 Apply plus_cancel_less with z:=[--]x1.
 Step_Rat_less_lft [--]x1.
 Step_Rat_less_rht [--]([--]e1).
 Apply min_resp_less.
 Assumption.

 Apply plus_cancel_less with z:=[--]x2.
 Step_Rat_less_lft [--]x2.
 Step_Rat_less_rht [--]([--]e2).
 Apply min_resp_less.
 Assumption.

 Step_Rat_less_lft (x1[*]x2[+]e1[*]e2)[+](x1[*]e2[+]e1[*]x2).
 Step_Rat_less_rht (x1[*]x2[+]e1[*]e2)[+](e1[*]e2[+]e1[*]e2).
 Apply plus_resp_less_lft.
 Apply plus_resp_less_both.
 Apply mult_resp_less.
 Assumption.
 Assumption.
 Apply mult_resp_less_lft.
 Assumption.
 Assumption.

 Apply plus_cancel_less with z:=[--](x1[*]x2).
 Step_Rat_less_lft Zero::R.
 Step_Rat_less_rht [--](x1[*]x2)[+](e1[*]e2)[+]((e1[*]e2)[+](e1[*]e2)).
 Apply less_transitive_unfolded with
   	 y:= [--](x1[*]x2)[+](e1[*]e2)[+]((x1[*]e2)[-](e1[*]x2)).
 Step_Rat_less_rht (e1[+]x1)[*](e2[-]x2).
 Apply mult_resp_pos.
 Apply plus_cancel_less with z:=[--]x1.
 Step_Rat_less_lft [--]x1.
 Step_Rat_less_rht [--]([--]e1).
 Apply min_resp_less.
 Assumption.

 Apply plus_cancel_less with z:=x2.
 Step_Rat_less_lft x2.
 Step_Rat_less_rht e2.
 Assumption.

 Apply plus_resp_less_lft.
 Step_Rat_less_lft  x1[*]e2[+]([--]e1[*]x2).
 Apply plus_resp_less_both.
 Apply mult_resp_less.
 Assumption.
 Assumption.
 Step_Rat_less_lft e1[*]([--]x2).
 Apply mult_resp_less_lft.
 Step_Rat_less_rht [--]([--]e2).
 Apply min_resp_less.
 Assumption.
 Assumption.

 Apply AbsSmall_pos with e:=e2 x:=x2.
 Assumption.
 Apply AbsSmall_pos with e:=e1 x:=x1.
 Assumption.
Qed.


(* Begin_Tex_Verb *)
Lemma AbsSmall_cancel_mult:(e,x,z:R)(Zero[:<]z)->
			(AbsSmall e[*]z x[*]z)->(AbsSmall e x).
(* End_Tex_Verb *)
 Unfold absSmall.
 Intros.
 Elim H0.
 Intros.
 Split.
 Apply mult_cancel_less with z:=z.
 Assumption.
 Step_Rat_less_lft [--](e[*]z).
 Assumption.
 Apply mult_cancel_less with z:=z.
 Assumption.
 Assumption.
Qed.

End AbsSmall_properties.


(* Tex_Prose
\subsection{Properties of one over successor}
\begin{convention}
Let \verb!R! be an ordered field.
\end{convention}
*)

(* Begin_Tex_Verb *)
Definition one_div_succ := [R:COrdField][i:nat](One [/] (Snring R i))
                        :  (R:COrdField)(nat->R).
(* End_Tex_Verb *)

Syntactic Definition One_div_succ := (one_div_succ ?).

Section One_div_succ_properties.

Variable R:COrdField.

(* Begin_Tex_Verb *)
Lemma one_div_succ_resp_leEq :
      (m,n:nat)(le m n)->((one_div_succ R n) [:<=] (one_div_succ R m)).
(* End_Tex_Verb *)
Unfold one_div_succ. Unfold Snring. Intros.
Apply recip_resp_leEq.
Apply pos_nring_S.
Apply nring_leEq.
Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma one_div_succ_pos : (i:nat)((Zero::R) [:<] (One_div_succ i)).
(* End_Tex_Verb *)
Intro.
Unfold one_div_succ.
Unfold Snring.
Apply recip_resp_pos.
Apply nring_pos.
Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma One_div_succ_resp_less : (i,j:nat)(lt i j)->
                              ((cof_less R) (One_div_succ j) (One_div_succ i)).
(* End_Tex_Verb *)
Intros.
Unfold one_div_succ. Unfold Snring. Intros.
Apply recip_resp_less.
Apply pos_nring_S.
Apply nring_less.
Auto with arith.
Qed.

(* Begin_Tex_Verb *)
Lemma One_div_succ_resp_leEq : (i,j:nat)(le i j)->
                              ((leEq R) (One_div_succ j) (One_div_succ i)).
(* End_Tex_Verb *)
Intros.
Apply one_div_succ_resp_leEq; Auto.
Qed.

End One_div_succ_properties.

(* Tex_Prose
\subsection{Properties of Half}
*)

(* We want the Syntactic Definition of Half to be globally visible, so it
   needs to be outside a section. Therefore the Coq Section starts a bit
   later than the Latex section *)

(* Begin_Tex_Verb *)
Definition half := [R:COrdField]One::R[/]TwoNZ.
(* End_Tex_Verb *)

Syntactic Definition Half := (half ?).

Syntax constr level 1:
  half_constant [<<(half $_)>>] -> ["Half"].

Section Half_properties.

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}
\end{convention}
*)

Variable R : COrdField.

(* Begin_Tex_Verb *)
Lemma half_1 : (Half::R)[*]Two [=] One.
(* End_Tex_Verb *)
Unfold half.
Apply div_1.
Qed.
Hints Resolve half_1 : algebra.

(* Begin_Tex_Verb *)
Lemma pos_half : ((Zero::R) [:<] Half).
(* End_Tex_Verb *)
Apply mult_cancel_pos_lft with (Two::R).
Apply (pos_wd R Half[*]Two One).
Exact half_1.
Apply pos_one.
Apply pos_two.
Qed.

(* Begin_Tex_Verb *)
Lemma half_1' : (x:R)((x[*]Half)[*]Two [=] x).
(* End_Tex_Verb *)
Intros.
Unfold half.
Rational.
Qed.

(* Begin_Tex_Verb *)
Lemma half_2 : (Half::R)[+]Half [=] One.
(* End_Tex_Verb *)
Unfold half.
Rational.
Qed.

(* Begin_Tex_Verb *)
Lemma half_lt1 : (Half::R)[:<] One.
(* End_Tex_Verb *)
Step_less_rht Half[+]Half::R.
Step_Rat_less_lft Half::R[+]Zero.
Apply plus_resp_less_lft.
Exact pos_half.
Exact half_2.
Qed.

(* Begin_Tex_Verb *)
Lemma half_3 : (x:R)(Zero [:<] x) -> (Half[*]x [:<] x).
(* End_Tex_Verb *)
Intros.
Step_less_rht One[*]x.
Apply mult_resp_less; Auto.
Exact half_lt1.
Qed.

End Half_properties.
Hints Resolve half_1 half_1' half_2 : algebra.


(* Begin_SpecReals *)

Section OrdField_Cauchy.

(* Tex_Prose
\subsection{Cauchy}
*)

(* Tex_Prose
\begin{convention} Let {\tt R} be a {\tt COrdField}.
\end{convention}
*)
Variable R : COrdField.

Implicit Arguments On.
(* Begin_Tex_Verb *)
Definition Cauchy_prop [g:nat -> R]: Prop :=
   (e:R)(Zero [:<] e) -> (Ex [N:nat](m:nat)(le N m)
			   -> (AbsSmall e ((g m)[-](g N)))).
(* End_Tex_Verb *)
Implicit Arguments Off.

(* Begin_Tex_Verb *)
(* Def. CauchyP, Build_CauchyP *)
(* Should be defined in terms of CauchyP *)
(* Implicit arguments turned off, because Coq makes a mess of it in combination
   with the coercions *)
Record CauchySeq : Set :=
  {CS_seq  :> nat -> R;
   CS_proof: (Cauchy_prop CS_seq)
  }.
(* End_Tex_Verb *)

(* Begin_Tex_Verb *)
Definition seqLimit [seq:nat->R; lim:R]: Prop :=
   (e:R)(Zero [:<] e) -> (Ex [N:nat](m:nat)(le N m)
			   -> (AbsSmall e ((seq m)[-]lim))).
(* End_Tex_Verb *)

End OrdField_Cauchy.

Syntactic Definition SeqLimit := (seqLimit ?).

(* End_SpecReals *)



Lemma CS_seq_plus:(R:COrdField; f,g:nat->R)(Cauchy_prop f)->(Cauchy_prop g)->
	(Cauchy_prop [m:nat]((f m)[+](g m))).
 Intros.
 Unfold Cauchy_prop.
 Intros.
 Unfold Cauchy_prop in H.
 Unfold Cauchy_prop in H0.
 Cut ((EX N:nat | (m:nat)(le N m)->(absSmall R e[/]FourNZ (f m)[-](f N)))).
 Intro.
 Cut ((EX N:nat | (m:nat)(le N m)->(absSmall R e[/]FourNZ (g m)[-](g N)))).
 Intro.
 Case H2.
 Intros N1 H21.
 Case H3.
 Intros N2 H31.
 Exists (plus N1 N2).
 Intros.
 Step_Rat_AbsSmall_lft (e[/]TwoNZ)[+](e[/]TwoNZ).
 Step_Rat_AbsSmall_rht  (f m)[-](f (plus N1 N2))[+]((g m)[-](g (plus N1 N2))).
 Apply AbsSmall_plus.

  Step_Rat_AbsSmall_rht ((f m)[-](f N1))[+]((f N1)[-](f (plus N1 N2))).
  Step_Rat_AbsSmall_lft (e[/]FourNZ)[+](e[/]FourNZ).
  Apply AbsSmall_plus.
  Apply H21.
  Apply le_trans with m:=(plus N1 N2).
  Apply le_plus_l.
  Assumption.
  Apply AbsSmall_minus.
  Apply H21.
  Apply le_plus_l.


  Step_Rat_AbsSmall_rht ((g m)[-](g N2))[+]((g N2)[-](g (plus N1 N2))).
  Step_Rat_AbsSmall_lft (e[/]FourNZ)[+](e[/]FourNZ).
  Apply AbsSmall_plus.
  Apply H31.
  Apply le_trans with m:=(plus N1 N2).
  Apply le_plus_r.
  Assumption.
  Apply AbsSmall_minus.
  Apply H31.
  Apply le_plus_r.

 Apply H0.
  Apply mult_cancel_less with z:=Four::R.
  Apply nring_pos.
  Apply lt_O_Sn.
  Step_Rat_less_lft Zero::R.
  Step_Rat_less_rht e.
  Assumption.

 Apply H.
  Apply mult_cancel_less with z:=Four::R.
  Apply nring_pos.
  Apply lt_O_Sn.
  Step_Rat_less_lft Zero::R.
  Step_Rat_less_rht e.
  Assumption.
Qed.


Section Mult_AbsSmall.

Variable R : COrdField.

(* Begin_Tex_Verb *)
Lemma estimate_sqrt :
  (a:R)(Zero [:<] a) ->
    (EX e | (Zero [:<] e) /\ (e[^](2) [:<] a)).
(* End_Tex_Verb *)
Intros.
Cut One [:<] Two::R. Intro.
Elim (cotrans_less_unfolded ? One Two H0 a); Intros.
Exists One::R.
Split. Apply pos_one.
Step_less_lft One::R. Auto. Auto.
Exists Half[*]a.
Cut Zero [:<] Half[*]a. Intro.
Split. Apply mult_resp_pos. Apply pos_half. Auto.
Step_less_lft (Half[*]a)[*](Half[*]a).
Apply less_wdr with (Half[*]Two)[*](One[*]a).
Apply mult_resp_less_both. Apply less_leEq. Auto.
Apply mult_resp_less_lft. Auto. Apply pos_half. Apply less_leEq. Auto.
Apply mult_resp_less. Unfold half. Apply shift_div_less. Apply pos_two.
Step_less_rht Two::R. Auto.
Auto.
Unfold half. Rational.
Apply mult_resp_pos. Apply pos_half. Auto.
Step_less_lft Zero[+]One::R. Step_Rat_less_rht One[+]One::R.
Apply plus_resp_less_rht. Apply pos_one.
Qed.

(* Begin_Tex_Verb *)
Lemma estimate_sqrt' :
  (a,b:R)(Zero [:<] a) -> (Zero [:<] b) ->
    (EX e | (Zero [:<] e) /\ (a[*]e[^](2) [:<] b)).
(* End_Tex_Verb *)
Intros.
Cut a [#] Zero. Intro.
Cut Zero [:<] b[/]a[//]H1. Intro.
Elim (estimate_sqrt ? H2). Intro e. Intros. Exists e.
Elim H3. Intros.
Split. Auto.
Step_less_lft e[^](2)[*]a.
Apply shift_mult_less with H1.
Auto.
Auto.
Apply div_resp_pos; Auto.
Apply Greater_imp_ap. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_abssmall'_rht :
  (x,y,C:R)(Zero [:<] C) ->
  ([--]C [:<] x) -> (x [:<] C) -> ([--]C [:<] y) -> (y [:<] C) ->
    (x[*]y [:<] Three[*]C[^](2)).
(* End_Tex_Verb *)
Intros.
Step_less_lft Zero[+]x[*]y. Apply shift_plus_less.
Apply less_transitive_unfolded with (C[+]x)[*](C[-]y).
Apply mult_resp_pos.
Apply shift_less_plus. Step_less_lft [--]x. Step_less_rht [--][--]C.
Apply min_resp_less. Auto.
Apply shift_less_minus. Step_less_lft y. Auto.
Step_Rat_less_lft C[^](2)[-]x[*]y[+]C[*](x[-]y).
Step_Rat_less_rht C[^](2)[-]x[*]y[+]C[*](C[-][--]C).
Apply plus_resp_less_lft.
Apply mult_resp_less_lft.
Apply minus_resp_less_leEq.
Apply less_leEq.
Auto. Auto. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_abssmall_rht :
  (x,y,X,Y:R)(Zero [:<] X) -> (Zero [:<] Y) ->
  ([--]X [:<] x) -> (x [:<] X) -> ([--]Y [:<] y) -> (y [:<] Y) ->
    (x[*]y [:<] X[*]Y).
(* End_Tex_Verb *)
Intros.
Elim (estimate_sqrt' Three X[*]Y (pos_three ?) (mult_resp_pos ? X Y H H0)).
Intro eps. Intros. Elim H5. Intros.
Cut [--]eps [:<] Zero. Intro.
Elim (cotrans_less_unfolded ? Zero eps H6 x); Intros.
Apply less_transitive_unfolded with x[*]Y.
Apply mult_resp_less_lft. Auto. Auto.
Apply mult_resp_less. Auto. Auto.
Elim (cotrans_less_unfolded ? [--]eps Zero H8 x); Intros.
Elim (cotrans_less_unfolded ? Zero eps H6 y); Intros.
Apply less_transitive_unfolded with X[*]y.
Apply mult_resp_less. Auto. Auto.
Apply mult_resp_less_lft. Auto. Auto.
Elim (cotrans_less_unfolded ? [--]eps Zero H8 y); Intros.
Apply less_transitive_unfolded with Three[*]eps[^](2).
Apply mult_abssmall'_rht; Auto.
Auto.
Step_Rat_less_lft [--]x[*][--]y.
Step_Rat_less_rht [--][--]X[*][--][--]Y.
Apply less_transitive_unfolded with [--][--]X[*][--]y.
Apply mult_resp_less. Apply min_resp_less. Auto.
Step_less_lft [--]Zero::R. Apply min_resp_less. Auto.
Apply mult_resp_less_lft. Apply min_resp_less. Auto.
Step_less_rht X. Auto.
Step_Rat_less_lft [--]x[*][--]y.
Step_Rat_less_rht [--][--]X[*][--][--]Y.
Apply less_transitive_unfolded with [--]x[*][--][--]Y.
Apply mult_resp_less_lft. Apply min_resp_less. Auto.
Step_less_lft [--]Zero::R. Apply min_resp_less. Auto.
Apply mult_resp_less. Apply min_resp_less. Auto.
Step_less_rht Y. Auto.
Step_less_rht [--]Zero::R. Apply min_resp_less. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_abssmall_lft :
  (x,y,X,Y:R)(Zero [:<] X) -> (Zero [:<] Y) ->
  ([--]X [:<] x) -> (x [:<] X) -> ([--]Y [:<] y) -> (y [:<] Y) ->
    ([--](X[*]Y) [:<] x[*]y).
(* End_Tex_Verb *)
Intros.
Step_Rat_less_rht [--]([--]x[*]y).
Apply min_resp_less.
Apply mult_abssmall_rht; Auto.
Apply min_resp_less. Auto.
Step_Rat_less_rht [--][--]X.
Apply min_resp_less. Auto.
Qed.

(* Begin_Tex_Verb *)
Lemma mult_abssmall :
  (x,y,X,Y:R)(AbsSmall X x) -> (AbsSmall Y y) -> (AbsSmall X[*]Y x[*]y).
(* End_Tex_Verb *)
Unfold absSmall.
Intros.
Elim H. Intros. Elim H0. Intros.
Cut Zero [:<] X. Intro. Cut Zero [:<] Y. Intro.
Split.
Apply mult_abssmall_lft; Auto.
Apply mult_abssmall_rht; Auto.
Apply mult_cancel_pos_rht with Two::R.
Step_Rat_less_rht Y[-][--]Y. Apply shift_less_minus. Step_less_lft [--]Y.
Apply less_transitive_unfolded with y; Auto.
Apply pos_two.
Apply mult_cancel_pos_rht with Two::R.
Step_Rat_less_rht X[-][--]X. Apply shift_less_minus. Step_less_lft [--]X.
Apply less_transitive_unfolded with x; Auto.
Apply pos_two.
Qed.

End Mult_AbsSmall.


Section Mult_Continuous.

Variable R : COrdField.

Lemma smaller :
  (x,y:R)(Zero [:<] x) -> (Zero [:<] y) ->
    (EX z | (Zero [:<] z) /\ (z [:<] x) /\ (z [:<] y)).
Intros.
Elim (cotrans_less_unfolded ??? (half_3 ?? H) y); Intro.
Exists Half[*]x.
Split. Apply mult_resp_pos. Apply pos_half. Auto.
Split. Apply half_3. Auto. Auto.
Exists Half[*]y.
Cut Half[*]y [:<] y. Intro.
Split. Apply mult_resp_pos. Apply pos_half. Auto.
Split. Apply less_transitive_unfolded with y. Auto. Auto.
Auto.
Apply half_3. Auto.
Qed.

Lemma estimate_abs : (x:R)(EX X | (AbsSmall X x)).
Intros.
Unfold absSmall.
Cut x [:<] x[+]One. Intro.
Elim (cotrans_less_unfolded ? x x[+]One H [--]x); Intro.
Exists [--]x[+]One.
Split.
Step_less_rht [--][--]x. Apply min_resp_less.
Apply less_plusOne.
Apply less_transitive_unfolded with [--]x. Auto. Apply less_plusOne.
Exists x[+]One.
Split.
Step_less_rht [--][--]x. Apply min_resp_less. Auto. Auto.
Apply less_plusOne.
Qed.

Lemma mult_contin :
  (x,y,e:R)(Zero [:<] e) ->
    (EX c | (Zero [:<] c) /\ (EX d | (Zero [:<] d) /\
      (x',y':R)(AbsSmall c x[-]x') -> (AbsSmall d y[-]y') ->
        (AbsSmall e x[*]y[-]x'[*]y'))).
Intros.
Cut Zero [:<] e[/]TwoNZ. Intro.
Elim (estimate_abs x). Intro X. Intros.
Elim (estimate_abs y). Intro Y. Intros.
Cut Zero [:<] Y. Intro.
Cut Y [#] Zero. Intro.
Exists (e[/]TwoNZ)[/]Y[//]H4.
Split.
Apply div_resp_pos. Auto. Auto.
Cut Zero [:<] X[+](e[/]TwoNZ)[/]Y[//]H4. Intro.
Cut X[+](e[/]TwoNZ)[/]Y[//]H4 [#] Zero. Intro.
Exists (e[/]TwoNZ)[/](X[+](e[/]TwoNZ)[/]Y[//]H4)[//]H6.
Split.
Apply div_resp_pos. Auto. Auto.
Intros.
Apply AbsSmall_wd_rht_unfolded with (x[-]x')[*]y[+]x'[*](y[-]y').
Apply AbsSmall_eps_div_two.
Apply AbsSmall_wd_lft_unfolded with ((e[/]TwoNZ)[/]Y[//]H4)[*]Y.
Apply mult_abssmall; Auto.
Rational.
Apply AbsSmall_wd_lft_unfolded with
  (X[+](e[/]TwoNZ)[/]Y[//]H4)[*]
    ((e[/]TwoNZ)[/](X[+](e[/]TwoNZ)[/]Y[//]H4)[//]H6).
Apply mult_abssmall; Auto.
Apply AbsSmall_wd_rht_unfolded with x[+](x'[-]x).
Apply AbsSmall_plus; Auto. Apply AbsSmall_minus. Auto.
Rational.
Rational.
Rational.
Apply Greater_imp_ap. Auto.
Apply plus_resp_pos; Auto.
Apply AbsSmall_pos with x. Auto.
Apply div_resp_pos; Auto.
Apply Greater_imp_ap. Auto.
Apply AbsSmall_pos with y. Auto.
Apply div_resp_pos. Apply pos_two. Auto.
Qed.

End Mult_Continuous.


