#load "unix.cma";;

let verbose_steps = ref true;;
let strict_checking = ref 2;;
let with_valid = ref true;;
let show_extra_goals = ref true;;
let stop_here = ref true;;
let clean_at_end = ref true;;
let run_at_qed = ref true;;
let report_time = ref false;;
let extra_lines = ref 80;;
let window_width = ref 80;;

let proof_filename = ref "/tmp/hol_light_proof";;
let goals_filename = ref "/tmp/hol_light_goals";;
let command_filename = ref "/tmp/hol_light_step_command";;
let pid_filename = ref "/tmp/hol_light_step_pid";;

type proof_step =
| Text_step
| Prove_step
| Let_step
| Tactic_step
| Then_step
| Thenl_step
| Lbra_step
| Semi_step
| Rbra_step
| Qed_step;;

type proof_state =
| Outside_proof_state
| Before_tactic_state of goal list * (goal list * goal list) list
| After_tactic_state of goal list * (goal list * goal list) list
| After_thenl_state of goal list * (goal list * goal list) list
| No_goals_left_state of goal list * (goal list * goal list) list;;

let the_command = ref "";;
let the_proof = ref "";;
let the_tail = ref "";;
let the_char = ref (-1);;
let the_steps = ref [];;
let the_target = ref (-1);;
let the_position = ref 0;;
let the_state = ref Outside_proof_state;;
let the_error = ref "";;
let the_cache = ref [];;

unset_jrh_lexer;;
let system_ok = Unix.WEXITED 0;;
let goals_flags = [Unix.O_WRONLY; Unix.O_CREAT; Unix.O_TRUNC];;
let hup_handler = ref (fun () -> print_string "hup_handler\n");;
Sys.signal Sys.sighup (Sys.Signal_handle (fun _ -> !hup_handler ()));;
set_jrh_lexer;;

let strip_control s =
  if String.contains s '\n' or String.contains s '\t' then
    let s = String.copy s in
    for n = 0 to String.length s - 1 do
      match String.get s n with
      | '\n' | '\t' -> String.set s n ' '
      | _ -> ()
    done;
    s
  else s;;

let rec single_space s =
  let s = strip_control s in
  let m = String.length s in
  let rec find_dup n =
    if n >= m then raise Not_found else
    let n' = String.index_from s n ' ' in
    let n'' = n' + 1 in
    if n'' < m & String.get s n'' = ' ' then n' else
    find_dup n'' in
  let rec find_nonspace n =
    if n >= m then raise Not_found else
    if String.get s n = ' ' then find_nonspace (n + 1) else n in
  let rec collect s' n =
    try let n' = find_dup n in
      try let n'' = find_nonspace (n' + 2) in
        collect (s'^" "^String.sub s n (n' - n)) n''
      with Not_found -> s'^" "
    with Not_found -> s'^" "^String.sub s n (m - n) in
  try let n' = find_dup 0 in
    try let n'' = find_nonspace (n' + 2) in
      collect (String.sub s 0 n') n''
    with Not_found -> String.sub s 0 n'^" "
  with Not_found -> s;;

let steps_of_strings s x l =
  let update_depth t d =
    match t with
    | "(" | "[" -> d + 1
    | ")" | "]" when d > 0 -> d - 1
    | _ -> d in
  let early n = if x < n then 0 else 1 in
  let step_of p n n0 n' =
    let s = String.sub s n (n0 - n) in
    ((p,(if p = Text_step then s else single_space s),(n,n0,n')),early n0) in
  let grow a r' r =
    if r' = [] then r else
    let (_,n0,n') = hd r' in
    let (n,_,_) = last r' in
    (step_of (if a = 1 then Tactic_step else Text_step) n n0 n')::r in
  let rec steps_of_strings a d r' r l =
    match l with
    | [] -> rev (grow a r' r)
    | (t,(n,n0,n'))::l' ->
         match a with
         | 0 ->
            (match l with
             | ("prove",(n,_,_))::("(",_)::(t,_)::(",",(_,n0,n'))::l' (* ) *)
                   when String.get t 0 = '`' ->
                 steps_of_strings 1 0 []
                   (((Prove_step,
                      single_space (String.sub t 1
                        (String.index_from t 1 '`' - 1)),
                      (n,n0,n')),early n0)::(grow a r' r)) l'
             | _ -> steps_of_strings a 0 ((n,n0,n')::r') r l')
         | 1 when d = 0 ->
             if t = "let" then
               steps_of_strings 2 0 [n,n0,n'] (grow a r' r) l' else
            (try
               let p =
                 match t with
                 | "THEN" -> Then_step
                 | "THENL" -> Thenl_step
                 | "[" when r' = [] -> Lbra_step
                 | ";" -> Semi_step
                 | "]" -> Rbra_step (* ( *)
                 | ")" -> Qed_step
                 | _ -> raise Not_found in
               steps_of_strings (if p = Qed_step then 0 else 1) 0
                 [] ((step_of p n n0 n')::(grow a r' r)) l'
             with Not_found ->
               steps_of_strings a (update_depth t d) ((n,n0,n')::r') r l')
         | 2 when d = 0 & t = "in" & r' <> [] ->
             let (_,_,nx) = last r' in
             let (_,ny,_) = hd r' in
             let s = single_space (String.sub s nx (ny - nx)) in
             steps_of_strings 1 0 []
               (((Let_step,s,(nx,n0,n')),early n0)::r) l'
         | _ -> steps_of_strings a (update_depth t d) ((n,n0,n')::r') r l' in
  steps_of_strings 0 0 [] [] l;;

let remove_marker x s' s =
  (* s' should have just one '|', and it should be the first character *)
  let m' = String.length s' in
  let m = String.length s in
  let rec clean_rec x s0 n0 n =
    try
      let n' = String.index_from s n '|' in
      let n'' = n' + m' in
      if n'' > m then ((s0^String.sub s n0 (m - n0)),x) else
      if String.sub s n' m' <> s' then clean_rec x s0 n0 (n' + 1) else
      let s0' = s0^String.sub s n0 (n' - n0) in
      let x' = String.length s0' in
      let x'' =
        if x >= x' + m' then x - m' else
        if x >= x' then x' else
        if x >= 0 then x else x' in
      clean_rec x'' s0' n'' n''
    with Not_found -> ((s0^String.sub s n0 (m - n0)),x) in
  try
    let _ = String.index s '|' in
    clean_rec x "" 0 0
  with Not_found -> (s,x);;

let remove_markers s =
  let (s,x) = remove_marker (-1) "|@" s in
  let (s,x) = remove_marker x "|#" s in
  (s,x);;

let split_steps s x =
  let m = String.length s in
  let idchar c =
    'a' <= c & c <= 'z' or 'A' <= c & c <= 'Z' or c = '_' or c = '\'' in
  let rec skip_id n =
    if n >= m then n else
    if idchar (String.get s n) then skip_id (n + 1) else n in
  let skip_term n =
    try String.index_from s n '`' + 1 with Not_found -> m in
  let rec skip_comment n k =
    if k = 0 then n else
    try
      let n' = String.index_from s n '*' in
      if n' > n & String.get s (n' - 1) = '(' then
        skip_comment (n' + 1) (k + 1)
      else if n' + 1 < m & String.get s (n' + 1) = ')' then
        skip_comment (n' + 2) (k - 1)
      else skip_comment (n' + 1) k
    with Not_found -> m in
  let rec find_next n0 n b w =
    if n >= m then (w,n0,n) else
    let n' = n + 1 in
    let c = String.get s n in
    match c with
    | '(' when n' < m & String.get s n' = '*' ->
        find_next n0 (skip_comment (n + 2) 1) b w
    | '`' -> if b then (w,n0,n) else
        let n'' = skip_term n' in
        find_next n'' n'' true (String.sub s n (n'' - n))
    | ' ' | '\n' | '\t' -> find_next n0 n' b w
    | _ when idchar c -> if b then (w,n0,n) else
        let n'' = skip_id n' in
        find_next n'' n'' true (String.sub s n (n'' - n))
    | _ -> if b then (w,n0,n) else
        find_next n' n' true (String.sub s n 1) in
  let rec split_rec n =
    if n >= m then [] else
    let (w,n0,n') = find_next n n false "" in
    (w,(n,n0,n'))::split_rec n' in
  let l = steps_of_strings s x (split_rec 0) in
  map fst l,if x < 0 then x else itlist (+) (map snd l) 0;;

let exec_phrase b s =
  let lexbuf = Lexing.from_string s in
  let ok = Toploop.execute_phrase b Format.std_formatter
    (!Toploop.parse_toplevel_phrase lexbuf) in
  Format.pp_print_flush Format.std_formatter ();
  (ok,
   let i = lexbuf.Lexing.lex_curr_pos in
   String.sub lexbuf.Lexing.lex_buffer
     i (lexbuf.Lexing.lex_buffer_len - i));;

let exec_tactic_out = ref ALL_TAC;;

let exec_tactic s =
  try
    let ok,rst = exec_phrase false
      ("let exec_phrase_out = exec_tactic_out := (("^s^") : tactic);;") in
    if not ok or rst <> "" then raise Noparse;
    !exec_tactic_out
  with _ -> raise Noparse;;

exception Step_error of string;;
let step_error s = raise (Step_error s);;

let take_step verbose step state =
  match step with
  | Text_step,_,_ ->
     (match state with
      | Outside_proof_state -> state
      | _ -> step_error "text in proof")
  | Prove_step,s,_ ->
      if state = Outside_proof_state then
       ((if verbose then print_string ("  ## `"^s^"`\n"));
        Before_tactic_state ([[],parse_term s],[]))
      else step_error "nested proofs"
  | Let_step,s,_ ->
     (match state with
      | Before_tactic_state _ ->
         (try
            (if verbose then print_string ("  ### "^s^"\n"));
            let _ = exec_phrase false ("let "^s^";;") in
            state
          with _ -> step_error "bad let")
      | _ -> step_error "let in wrong place")
  | Tactic_step,s,_ ->
     (match state with
      | Before_tactic_state (gs,rest) ->
          if !strict_checking >= 2 & gs = [] then
            step_error "no subgoals left" else
         (try let tac = exec_tactic s in
            let tac = if !with_valid then VALID tac else tac in
            (if verbose then print_string ("  # "^s^"\n"));
            After_tactic_state
              ((flat (map ((fun (_,gs',_) -> gs') o tac) gs)),rest)
          with
          | Noparse -> step_error "not a valid tactic"
          | Failure s -> step_error s)
      | No_goals_left_state _ -> step_error "no goals left in THENL list"
      | _ -> step_error "tactic in wrong place")
  | Then_step,_,_ ->
     (match state with
      | After_tactic_state (gs,rest) ->
          if !strict_checking >= 2 & gs = [] then
            step_error "no subgoals left" else
          Before_tactic_state (gs,rest)
      | _ -> step_error "THEN in wrong place")
  | Thenl_step,_,_ ->
     (match state with
      | After_tactic_state (gs,rest) ->
          if !strict_checking >= 1 & gs = [] then
            step_error "no subgoals left" else
          After_thenl_state (gs,rest)
      | _ -> step_error "THENL in wrong place")
  | Lbra_step,_,_ ->
     (match state with
      | After_thenl_state (gs,rest) ->
          if gs = [] then No_goals_left_state (gs,rest) else
          Before_tactic_state ([hd gs],(([],tl gs)::rest))
      | _ -> step_error "[ in wrong place")
  | Semi_step,_,_ ->
     (match state with
      | After_tactic_state (gs,((lgs,rgs)::rest)) ->
          if !strict_checking >= 2 & gs <> [] then
            step_error "unsolved subgoals" else
          if rgs = [] then No_goals_left_state (lgs,rest) else
          Before_tactic_state ([hd rgs],(((lgs@gs),tl rgs)::rest))
      | After_tactic_state (_,[]) -> step_error "not in THENL list"
      | _ -> step_error "; in wrong place")
  | Rbra_step,_,_ ->
     (match state with
      | After_tactic_state (gs,((lgs,[])::rest)) ->
          if !strict_checking >= 3 & gs <> [] then
            step_error "unsolved subgoals" else
          After_tactic_state ((lgs@gs),rest)
      | After_tactic_state (_,((_,_)::_)) ->
          step_error "remaining goals in THENL list"
      | After_tactic_state (_,[]) -> step_error "not in THENL list"
      | No_goals_left_state (gs,rest) -> After_tactic_state (gs,rest)
      | _ -> step_error "] in wrong place")
  | Qed_step,_,(_,n,_) ->
     (match state with
      | After_tactic_state ([],[]) ->
         (if !run_at_qed then
           (try ignore (exec_phrase true (String.sub !the_proof 0 n^";;"))
            with _ -> ()));
          Outside_proof_state
      | After_tactic_state (_,[]) -> step_error "unsolved goals"
      | After_tactic_state _ -> step_error "proof not closed"
      | _ -> step_error "close bracket in wrong place");;

let rep_of_state state =
  let rec rep c n m s =
    match c with
    | [] -> string_of_int n^"/"^string_of_int m^" "^s
    | (l,r)::c' ->
        let l = length l and r = length r in
        rep c' n (m + l + r)
          ("("^(if l = 0 then "" else string_of_int l^"+")^s^
           (if r = 0 then "" else "+"^string_of_int r)^")") in
  match state with
  | Outside_proof_state -> ""
  | Before_tactic_state (l,c)
  | After_tactic_state (l,c)
  | After_thenl_state (l,c)
  | No_goals_left_state (l,c) ->
      let l = length l in
      rep c l l ("("^string_of_int l^")");;

let goals_of_proofstate state =
  let rec goals_of c l l' =
    match c with
    | [] -> l,l'
    | (l1,l2)::c' -> goals_of c' l (l'@l2@l1) in
  match state with
  | Outside_proof_state -> [],[]
  | Before_tactic_state (l,c)
  | After_tactic_state (l,c)
  | After_thenl_state (l,c)
  | No_goals_left_state (l,c) ->
      goals_of c l [];;

let rec print_goallist l =
  if l = [] then () else (print_goallist (tl l); print_goal (hd l));;

let print_proofstate state msg =
  let print_bar c = print_string (String.make !window_width c) in
  let l,l' = goals_of_proofstate state in
  if !show_extra_goals & l' <> [] then
   (print_goallist l';
    print_bar '-';
    print_string "\n";
    if l = [] then print_string "\n");
  print_goallist l;
  print_bar '-';
  print_string "\n";
  let r = rep_of_state state in
  let n = !window_width - String.length r - String.length msg - 1 in
  print_string (r^(if n < 3 then "\n" else String.make n ' ')^msg^"\n");;

let take_step' step state =
  let state' = take_step !verbose_steps step state in
  let step',_,_ = step in
 (match step' with
  | Prove_step | Tactic_step | Lbra_step | Semi_step | Rbra_step ->
      print_string "  ";
      print_string (rep_of_state state');
      print_string "\n"
  | _ -> ());
  state';;

let read_proof () =
  let f = Pervasives.open_in !proof_filename in
  let n = in_channel_length f in
  let s = String.create n in
  really_input f s 0 n;
  close_in f;
  let (s,x) = remove_markers s in
 (if !the_command = "here" & !stop_here then
   (the_proof := String.sub s 0 x;
    the_tail := String.sub s x (String.length s - x))
  else
   (the_proof := s;
    the_tail := ""));
  the_char := x;
  let (l,n) = split_steps !the_proof x in
  the_steps := l;
  the_target := n;
  the_position := -1;
  the_state := Outside_proof_state;
  the_error := "proof has not been run";;

let run_proof () =
  let rec run steps cache =
    if steps = [] or !the_position >= !the_target then
      (if cache <> [] then the_cache := rev cache@ !the_cache) else
    let step = hd steps in
    let state,cache' =
     (try if cache = [] then raise Not_found;
      	let (step',state') = hd cache in
      	if step <> step' then raise Not_found;
      	(state',tl cache)
      with Not_found -> (take_step !verbose_steps step !the_state,[])) in
    the_state := state;
    the_position := !the_position + 1;
    the_cache := (step,state)::!the_cache;
    run (tl steps) cache' in
  if !the_target >= 0 then
   (the_state := Outside_proof_state;
    the_position := 0;
    the_error := "";
    let cache = !the_cache in
    the_cache := [];
    run !the_steps (rev cache));;

let write_proof () =
  let rec at_end l =
    match l with
    | (Qed_step,_,_)::_ -> true
    | (Text_step,_,_)::l' -> at_end l'
    | _ -> false in
  let s' = if !the_position < 0 then !the_proof else
    let n = String.length !the_proof in
    if !the_position < 0 then !the_proof else
    let l,l' = chop_list !the_position !the_steps in
    if l' = [] & !clean_at_end & at_end (rev l) then !the_proof else
    let n' = (if l = [] then 0 else
      if l' = [] & !the_command = "here" & !stop_here then
        String.length !the_proof else
      let (_,_,(_,n',_)) = last l in n') in
    String.sub !the_proof 0 n'^"|#"^String.sub !the_proof n' (n - n') in
  let f = Pervasives.open_out !proof_filename in
  output_string f (if !the_tail = "" then s' else s'^ !the_tail);
  close_out f;;

let write_goals () =
  print_flush ();
  flush stdout;
  let fd0 = Unix.descr_of_out_channel stdout in
  let fd0' = Unix.dup fd0 in
  let fd1 = Unix.openfile !goals_filename goals_flags 0o644 in
  Unix.dup2 fd1 fd0;
  print_string (String.make !extra_lines '\n');
  print_proofstate !the_state !the_error;
  print_flush ();
  flush stdout;
  Unix.dup2 fd0' fd0;
  Unix.close fd1;
  Unix.close fd0';;

let here_stop () = ();;

let exec_stop () =
  the_target := length !the_steps;;

let active_step step =
  let x,_,_ = step in
  match x with
  | Prove_step | Tactic_step | Lbra_step | Semi_step | Rbra_step -> true
  | _ -> false;;

let next_stop () =
  let rec find_next n l =
    if l = [] then n else
    if active_step (hd l) then n + 1 else
    find_next (n + 1) (tl l) in
  let n = !the_target in
  if n < 0 then the_target := find_next 0 !the_steps else
  let l1,l2 = chop_list n !the_steps in
  if l2 <> [] then the_target := find_next n l2;;

let prev_stop () =
  let rec find_prev n l =
    if l = [] then n else
    if active_step (hd l) then n else
    find_prev (n - 1) (tl l) in
  let n = !the_target in
  if n < 0 then the_target := 0 else
  let l1,l2 = chop_list n !the_steps in
  if l1 <> [] then the_target := find_prev (n - 1) (tl (rev l1));;

let step_hol_light move_stop =
  the_error := "";
 (try
    read_proof ();
    move_stop ();
    run_proof ()
  with
  | Sys.Break -> the_error := "interrupted"
  | Step_error msg -> the_error := msg
  | Failure msg -> the_error := "failure: "^msg
  | Not_found -> the_error := "not found"
  | _ -> the_error := "unknown error");
  write_proof ();
  write_goals ();;

let run_rest () =
 (try
    read_proof ();
    write_proof ();
  with _ -> the_error := "cleaning proofs failed");
 (try
    if Toploop.use_file Format.std_formatter !proof_filename then
      the_error := "" else the_error := "error in ocaml"
  with _ -> the_error := "processing ocaml failed");
  write_goals ();;

hup_handler := fun () ->
  let starttime = Unix.gettimeofday() in
  (Format.pp_print_flush Format.std_formatter ();
  (try
     let f = Pervasives.open_in !command_filename in
     let s = input_line f in
     close_in f;
     the_command := s;
    (match s with
     | "here" -> step_hol_light here_stop
     | "exec" -> step_hol_light exec_stop
     | "next" -> step_hol_light next_stop
     | "prev" -> step_hol_light prev_stop
     | "proceed" -> run_rest ()
     | _ -> ())
   with _ -> ());
   Unix.unlink !command_filename;
   if !report_time then
     (print_float (Unix.gettimeofday() -. starttime); print_string "\n");
   Format.pp_print_flush Format.std_formatter ());;

let exit_proc = ref (fun () -> ());;

let server_up () =
  if Unix.system ("ps -p $$ -o ppid= | sed 's/ //g' > "^
    !pid_filename) <> system_ok then
  print_string "server_up failed\n";;

let server_down () =
  ignore (Unix.system ("rm -f "^ !pid_filename));;

server_up();;
exit_proc := server_down;;
at_exit (fun _ -> !exit_proc ());;


(*
let SNDCART_PASTECART_string = "let SNDCART_PASTECART = prove
 (`!x y. sndcart(pastecart (x:A^M) (y:A^N)) = y`,
  SIMP_TAC[pastecart; sndcart; CART_EQ; LAMBDA_BETA] THEN REPEAT STRIP_TAC THEN
  W(fun (_,w) -> MP_TAC (PART_MATCH (lhs o rand) LAMBDA_BETA (lhand w))) THEN
  ANTS_TAC THENL
   [REWRITE_TAC[DIMINDEX_FINITE_SUM] THEN MATCH_MP_TAC
     (ARITH_RULE `1 <= i /\ i <= b ==> 1 <= i + a /\ i + a <= a + b`) THEN
    ASM_REWRITE_TAC[];
    DISCH_THEN SUBST1_TAC THEN REWRITE_TAC[] THEN
    ASM_SIMP_TAC[ADD_SUB; ARITH_RULE `1 <= i ==> ~(i + a <= a)`]]);;
";;

let SNDCART_PASTECART_steps' =
 [Text_step,"let SNDCART_PASTECART =",(0,23,24);
  Prove_step,"!x y. sndcart(pastecart (x:A^M) (y:A^N)) = y",(24,79,82);
  Tactic_step,"SIMP_TAC[pastecart; sndcart; CART_EQ; LAMBDA_BETA]",
    (82,132,133);
  Then_step,"THEN",(133,137,138);
  Tactic_step,"REPEAT STRIP_TAC",(138,154,155);
  Then_step,"THEN",(155,159,162);
  Tactic_step,
    "W(fun (_,w) -> MP_TAC (PART_MATCH (lhs o rand) LAMBDA_BETA (lhand w)))",
    (162,232,233);
  Then_step,"THEN",(233,237,240);
  Tactic_step,"ANTS_TAC",(240,248,249);
  Thenl_step,"THENL",(249,254,258);
  Lbra_step,"[",(258,259,259);
  Tactic_step,"REWRITE_TAC[DIMINDEX_FINITE_SUM]",(259,291,292);
  Then_step,"THEN",(292,296,297);
  Tactic_step,
    "MATCH_MP_TAC (ARITH_RULE `1 <= i /\\ i <= b ==> 1 <= i + a /\\ i + a <= a + b`)",
    (297,379,380);
  Then_step,"THEN",(380,384,389);
  Tactic_step,"ASM_REWRITE_TAC[]",(389,406,406);
  Semi_step,";",(406,407,412);
  Tactic_step,"DISCH_THEN SUBST1_TAC",(412,433,434);
  Then_step,"THEN",(434,438,439);
  Tactic_step,"REWRITE_TAC[]",(439,452,453);
  Then_step,"THEN",(453,457,462);
  Tactic_step,"ASM_SIMP_TAC[ADD_SUB; ARITH_RULE `1 <= i ==> ~(i + a <= a)`]",
    (462,522,522);
  Rbra_step,"]",(522,523,523);
  Qed_step,")",(523,524,524);
  Text_step,";;",(524,526,527);
 ];;

let SNDCART_PASTECART_steps,_ = split_steps SNDCART_PASTECART_string 0;;

if SNDCART_PASTECART_steps <> SNDCART_PASTECART_steps' then
  failwith "example";;
*)

