package jason.client;

/**
 * <p>Title: Javacard As Secure Object Store</p>
 * <p>Description: Serializer for primitive data types</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: University of Twente</p>
 * @author Richard Brinkman
 * @version 1.0
 */
public class Appender {
	/** Indicates whether the length of an array should be stored as well */
	protected boolean storeLengths;
	/** Internal buffer */
	protected byte[] buffer;

	/**
	 * Makes a new Appender with the {@link #storeLengths} variable set to true.
	 */
	public Appender() {
		this(true);
	}

	/**
	 * Makes a new Appender.
	 * @param storeLengths Indicates whether the length of an array should be stored as well
	 */
	public Appender(boolean storeLengths) {
		this.storeLengths = storeLengths;
		buffer = new byte[0];
	}

	/**
	 * Build a bigger array containing the old array and the specified part of the
	 * given array. Subklasses of {@link Appender} only have to modify this method.
	 * @param b The data to add to the internal buffer
	 * @param offset Offset within <code>b</code>
	 * @param length Length of bytes to copy
	 */
	protected void appendIntern(byte[] b, int offset, int length) {
		byte[] newBuffer = new byte[buffer.length + length];
		System.arraycopy(buffer, 0, newBuffer, 0, buffer.length);
		System.arraycopy(b, offset, newBuffer, buffer.length, length);
		buffer = newBuffer;
	}

	/**
	 * Appends to whole array.
	 * @param b Data to be added
	 */
	protected void appendIntern(byte[] b) {
		appendIntern(b, 0, b.length);
	}

	/**
	 * Appends a single byte.
	 * @param b Data to be added
	 */
	public void append(byte b) {
		appendIntern(new byte[] {b});
	}

	/**
	 * Adds a single boolean.
	 * @param b Data to be added
	 */
	public void append(boolean b) {
		append((byte) (b ? 1 : 0));
	}

	/**
	 * Appends a single short.
	 * @param s Data to be added
	 */
	public void append(short s) {
		append((byte) (s & 0x00FF));
		append((byte) (s >> 8));
	}

	/**
	 * Appends only the specified part of the array.
	 * @param b Data to be added
	 * @param offset Offset within <code>b</code>
	 * @param length Length of bytes to be copied
	 */
	public void append(byte[] b, int offset, int length) {
		if (storeLengths)
			append((short) length);
		appendIntern(b, offset, length);
	}

	/**
	 * Appends the whole byte array.
	 * @param b Data to be added
	 */
	public void append(byte[] b) {
		append(b, 0, b.length);
	}

	/**
	 * Appends another appender (i.e. appending the internal buffer of the other
	 * Appender).
	 * @param appender The Appender to be appended
	 */
	public void append(Appender appender) {
		append(appender.getBuffer());
	}

	/**
	 * Gets the internal buffer. Note that it is the buffer itself and not a copy.
	 * @return The internal buffer.
	 */
	public byte[] getBuffer() {
		return buffer;
	}
}
